(* ::Package:: *)

(* :Title: Mend *)

(* :Author: Ekkehart Schlicht *)

(* :Summary:
   This package provides a function to correct a given time series for missing values and structural 
   breaks according to method described in 
   "Trend Extraction from Time Series with Structural Breaks and Missing Observations", 
   by Ekkehart Schlicht, Journal of the Japan Statistical Society
   Vol. 38 (2008), No. 2, pages 285-292, available at
   http://www.jstage.jst.go.jp/article/jjss/38/2/285/_pdf. 
*)

(* :Mathematica Version: 8.01 
   Program version 1 of April 24, 2011
   
*)




Mend::usage = "\!\(\*
StyleBox[\"Mend\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"[\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"\[Alpha]\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"breaks\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"]\",\nFontSlant->\"Italic\"]\) repairs the time series \!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\) by filling gaps with estimates and correcting for breakpoints.
The vector \!\(\*
StyleBox[\"breaks\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"=\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"{\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[SubscriptBox[\"bp\", \"1\"],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[SubscriptBox[\"bp\", \"2\"],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"...\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"}\",\nFontSlant->\"Italic\"]\) gives the breakpoints.

\!\(\*
StyleBox[\"Mend\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"[\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"]\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)returns a list \!\(\*
StyleBox[\"{\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"y\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"breaks\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"gaps\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"}\",\nFontSlant->\"Italic\"]\)
-   \!\(\*FormBox[
RowBox[{\"y\", \" \"}],
TraditionalForm]\)= {\!\(\*FormBox[SubscriptBox[\"y\", \"1\"],
TraditionalForm]\),\!\(\*FormBox[SubscriptBox[\"y\", \"2\"],
TraditionalForm]\),..\!\(\*FormBox[SubscriptBox[\"y\", \"T\"],
TraditionalForm]\)}  is the mended time series, 
-   \!\(\*
StyleBox[\"breaks\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"=\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"{\",\nFontSlant->\"Italic\"]\)\!\(\*FormBox[
RowBox[{
RowBox[{
StyleBox[
RowBox[{
RowBox[{SubscriptBox[\"break\", \"1\"], \",\", \" \", SubscriptBox[\"break\", \"2\"], \",\", \" \", \"..\"}], \"}\"}],\nFontSlant->\"Italic\"], \" \", \"gives\", \" \", \"the\", \" \", \"breaks\"}], \",\", \" \", \"with\"}],
TraditionalForm]\) \!\(\*
StyleBox[SubscriptBox[\"break\", \"i\"],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"=\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"{\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[FormBox[SubscriptBox[\"bp\", 
RowBox[{\"i\", \" \"}]],
TraditionalForm],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[FormBox[
RowBox[{\" \", SubscriptBox[\"d\", \"i\"]}],
TraditionalForm],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\")\",\nFontSlant->\"Italic\"]\) denoting the position \!\(\*
StyleBox[FormBox[SubscriptBox[\"bp\", 
RowBox[{\"i\", \" \"}]],
TraditionalForm],\nFontSlant->\"Italic\"]\)of break \!\(\*
StyleBox[\"i\",\nFontSlant->\"Italic\"]\)  and  \!\(\*
StyleBox[FormBox[SubscriptBox[\"d\", \"i\"],
TraditionalForm],\nFontSlant->\"Italic\"]\) denoting the dummy.
-   \!\(\*
StyleBox[\"gaps\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"=\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"{\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[FormBox[
RowBox[{
RowBox[{
RowBox[{SubscriptBox[\"gap\", \"1\"], \",\", \" \", SubscriptBox[\"gap\", \"2\"], \",\", \" \", \"..\"}], \"}\"}], \" \", 
StyleBox[\"gives\",\nFontSlant->\"Plain\"], 
StyleBox[\" \",\nFontSlant->\"Plain\"], 
StyleBox[\"the\",\nFontSlant->\"Plain\"], 
StyleBox[\" \",\nFontSlant->\"Plain\"], 
StyleBox[\"gaps\",\nFontSlant->\"Plain\"]}],
TraditionalForm],\nFontSlant->\"Italic\"]\) \!\(\*
StyleBox[SubscriptBox[\"gap\", \"i\"],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"=\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"{\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[FormBox[SubscriptBox[\"g\", 
RowBox[{\"i\", \" \"}]],
TraditionalForm],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[FormBox[
RowBox[{\" \", SubscriptBox[\"r\", \"i\"]}],
TraditionalForm],\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\")\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\), denoting that gap \!\(\*
StyleBox[\"i\",\nFontSlant->\"Italic\"]\) filled by replacement\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)\!\(\*SubscriptBox[
StyleBox[\"r\",\nFontSlant->\"Italic\"], 
RowBox[{\"i\", \" \"}]]\)."

GuessBreak::usage = "\!\(\*
StyleBox[\"GuessBreak\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"[\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"\[Alpha]\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"]\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\" \",\nFontSlant->\"Italic\"]\)returns a number that indicates the position of a possible structural break. It requires as an input the time series and the smoothing parameter \[Alpha]."

Criterion::usage = "\!\(\*
StyleBox[\"Criterion\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"[\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\",\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"\[Alpha]\",\nFontSlant->\"Italic\"]\)\!\(\*
StyleBox[\"]\",\nFontSlant->\"Italic\"]\) mends possible gaps in the time series \!\(\*
StyleBox[\"x\",\nFontSlant->\"Italic\"]\) and returns the value of the criterion that underlies the mending procedure. It requires as an input the time series and the smoothing parameter \[Alpha]."


Unprotect[Mend,Criterion,GuessGap];
Mend[x_,\[Alpha]_:100,bp_:{}]:=
Module[{ud,z,T,D,E,F,P,M,d},(
	
	Mend::singular="Singularity encountered, probably due to an overlap or bad combination of undefined data points and breaks.";
	On[Inverse::luc];On[Mend::singular];
	ud:= Flatten[Position[x,Null]];	
	z:=x/. Null ->0.;
	T=Length[x];
	If[bp!={},D=Transpose[Map[SparseArray[i_/;i>=#-> 1.0,{T}]&,bp]]];
	If[ud!={},E=SparseArray[MapIndexed[({#1,Last[#2]}->1.0)&,ud],{T,Length[ud]}]];
	If[ud!={}&&bp!={},F=(Transpose[ArrayFlatten[{Transpose[D],Transpose[E]},1]])];
	If[ud!={}&&bp=={},F=E];
	If[ud=={}&&bp!= {},F=D];
	If[ud!={}||bp!= {},
		P=SparseArray[{{i_,i_}->1,{i_,j_}/;j==i+1->-2,{i_,j_}/;j==i+2->1},{T-2,T}];
		M=Inverse[IdentityMatrix[T]+\[Alpha] Transpose[P].P];
		If[MatrixQ[F],trF=Transpose[F],trF=F];
		Check[d=-Inverse[trF.(IdentityMatrix[T]-M).F].trF.(IdentityMatrix[T]-M).z ,Message[Mend::singular];Abort[]];
	{z+F.d ,{Transpose[{bp,d[[1;;Length[bp]]]}],Transpose[{ud,d[[Length[bp]+1;;Length[d]]]}]}},
	{x,{},{}}])
];

Criterion[x_,\[Alpha]_:100]:=Module[{ud,z,T,E,F,P,M,d},
(
T=Length[x];
ud:= Flatten[Position[x,Null]];
P=SparseArray[{{i_,i_}->1,{i_,j_}/;j==i+1->-2,{i_,j_}/;j==i+2->1},{T-2,T}];	M=IdentityMatrix[T]-Inverse[IdentityMatrix[T]+\[Alpha] Transpose[P].P];

If[ud=={},
x.M.x,

z:=x/. Null ->0.;	
E=SparseArray[
		MapIndexed[({#1,Last[#2]}->1.0)&,ud],{T,Length[ud]}];
	
	If[MatrixQ[E],
	trE=Transpose[E],
	trE=E
	];
				
	d=-Inverse[trE.M.E].trE.M.z ;

(z+E.d ).M.(z+E.d )]
)]

GuessBreak[x_,\[Alpha]_:100,bp_:{}]:=Module[{x1,T, k,crit,b},(
	T=Length[x]; crit={Null };
	Do[	
		crit=Append[crit,Criterion[Mend[x,100,{k}][[1]]]],
		{k,3,T-1}];
	crit=Append[crit,Null];
	{b}=Flatten[Position[crit,Min[crit][[1]]]+1];b
)]


Protect[Mend,Criterion,GuessGap];



