// Version 5.0 of October 2004
// Version 5.7 of October 2005
// Version 6   of April 2021
// All rights reserved
// Published under the GNU Public License

// Program originally written 1985 in HP-Basic by Ekkehart Schlicht
// Ported to C by Robert Schlicht in 2002.
// Various additions by Ekkehart Schlicht.
// Helpful suggestions by Johannes Ludsteck.

// To compile with the Microsoft C compiler, use a command 
// like this one:
// "C:\Program Files\Microsoft C\bin\cl" /Ox /I "C:\Program Files\Microsoft C\include"
//		   vcc.c /link /subsystem:windows /libpath:"C:\Progra~1\Micros~2\lib"
//		   kernel32.lib user32.lib gdi32.lib comdlg32.lib


// To compile under Linux use a command like
// gcc -o vcc vcc.c -lm 
// and uncomment the following line:
// #define LINUX 1


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <time.h>

// Some auxiliary functions for handling vectors and matrices.
// ===========================================================
/*
 The function 'vector' creates a vector or changes its length.
 'x' points to the address of the vector, which is set to 0 when the
 vector has the length 0, and must be set to 0 before the function is
 function is called for the first time for a vector. 'length' is
 the new length of the vector, which will be stored. 'size' is the
 memory requirement of a single element in bytes. If there is not
 enough memory free, the program is aborted.
 The macro 'LENGTH' determines the length of a vector:
 */

	
#define LENGTH(x) ( !(x)? 0 : *((unsigned int*)(void*)(x) - 1) )

void vector(void ** x, unsigned int size, unsigned int length)
{
	unsigned int *ptr;
	ptr = !*x? 0 : (unsigned int*)*x - 1;			  		// Determine start of memory area
	if (!length) {if (ptr) free(ptr); *x = 0; return;} 		// Free memory if length=0 
	if (ptr && *ptr == length) return;				   		// Otherwise check if size is changed
	ptr = realloc(ptr, sizeof *ptr + length * size);   		// Change the size of the memory area
	if (!ptr) abort();								   		// Abort program in case of error
	*x = (void*)(ptr + 1);							   		// Set start of vector to second element
	*ptr = length;									   		// store vector length in the first element
}


// The same function for a matrix (= vector of vectors):

void matrix(void *** A, unsigned int size, unsigned int length, unsigned int width)
{
	unsigned int n;
	n = LENGTH(*A); 										 // store old matrix length
	while (n > length) vector(&(*A)[n], size, 0);	         // Delete superfluous lines
	vector((void**)A, sizeof **A, length);					 // Adjust matrix length
	while (n < length) (*A)[n++] = 0;						 // initialize additional rows
	for (n=0; n<length; n++) vector(&(*A)[n], size, width);  // Adjust row lengths
}


// The following two macros facilitate the use of vector and matrix:

#define VECTOR(x, length) vector((void**)&(x), sizeof *(x), (length))
#define MATRIX(A, length, width) matrix((void***)&(A), sizeof **(A), (length), (width))


// Definition of program options taken from the VC-Program (Windows version)
// =========================================================================

// Programm-Optionen (globale Variablen)
// slow		controls adjustment
// cov		flag for calculating covariance matrix
// filter	don't estimate variances, just filter with given variance ratios
// *check	indicates which variances to freeze (not adjust)
// convergence flag indicating convergence

 
// flag[i]=0: estimate time-varying coefficient for coefficient in column i+2
// flag[i]=1: keep smoothing weight (variance ratio r[i]) constant
// flag[i]=2: keep coefficient in column i+2 constant 


int slow, cov, *flag, filter, convergence,icalc;     // Flags
double smooth=0., s0calc=0., s0filter=0.;

static double **C;

// Main program
// ============

int main(int argc, char *argv[])
{
	void sol(double **x, double *y, double **A, double *r, double *z); //x, y and r are given, A is determined.
	void standarderror(double **C, double *as);
	void errormatrix(double **c, double *as);
	double var(double **x, double *y, double **A, double *r, double *s0, double *s, double *z);

	FILE *file;
	double **x=0, *y=0, **A=0, *as=0, **AS=0,*r=0, *z=0, *s=0, *sini=0, *rini=0,*rcalc=0,*rfilter=0,s0=0;
	double v, vcalc, precision;

	unsigned int T, n, i, j, t, arguments, understood, zerovar, negvar,  numeric, maxit;
	char c, name[1024], infile[1024], outfile[1024], helptext[1024], message[1024], **rname=0, **cname = 0;
	char * pEnd;

// Help text
		 strcpy(helptext, "\nvcc - a program for estimating time-varying coefficients\n");
		 strcat(helptext,"by Ekkehart Schlicht, version 6 \nPublished under the GNU Public License.\n");
		 strcat(helptext,"All rights reserved. Available at www.lrz.de/~ekkehart\n\n");
         strcat(helptext,"Usage: vcc. inputfile [switches] with:\n");
		 strcat(helptext,"   inputfile        path of input file (CSV format - see documentation)\n");
		 strcat(helptext,"	            Beginning with column 3, the first line gives the starting values\n");
		 strcat(helptext,"	            for the variance ratios. Negative variance ratios will be frozen to their\n");
		 strcat(helptext,"                    absolute values and not updated during iterations. Zero variance ratios\n");
		 strcat(helptext,"	            will be frozen close to zero.\n\n");
		 strcat(helptext,"   Switches         Function \n");
		 strcat(helptext,"   --h or --help    Show this help screen.\n");
		 strcat(helptext,"   --smooth n       Increase estimated variance ratios by a factor of 10^n.\n");
		 strcat(helptext,"                    Default is n=0 (no smoothing).\n");
		 strcat(helptext,"   --f or --filter  Don't estimate variances, just filter.\n");
		 strcat(helptext,"   --p n            Set precision 10^-n. Default is 5 which means 10^-5.\n");
		 strcat(helptext,"   --it n           Set maximum number of iterations to 10^n. Default is 10^5.\n");
		 strcat(helptext,"   --cov            Calculate the complete covariance matrix.\n");
		 strcat(helptext,"   --s or --slow    Slow down adjustment speed to improve numerical stability.\n");

// Evaluate command line
	
		 arguments=argc;
		 numeric=0;
		 precision=0.00001;
		 maxit=100000;

	if (argc==1) { printf("%s",helptext); return 0;};
	if (argc==2) { 
		if((strcmp(argv[1],"--h")==0)||(strcmp(argv[1],"--help")==0)) printf("%s",helptext); 
		else strcpy(infile,argv[1]);
		};
	if (argc > 2) { 
		
	       for(i=2; i<arguments;i++) {
				if(numeric){
				numeric=0;
				understood=1;
			   }
			   else {
				   understood=0;
				if (strcmp(argv[i],"--h")==0){printf("%s",helptext); understood=1;};		// help
				if (strcmp(argv[i],"--help")==0){printf("%s",helptext); understood=1;};	// help
				if (strcmp(argv[i],"--smooth")==0){									// set smooth 
				   smooth=atof(argv[i+1]);
				   smooth=pow(10.,smooth);
				   numeric=1;
				   understood=1; };
  				if (strcmp(argv[i],"--f")==0){understood=1;filter=1;};				// just filter
				if (strcmp(argv[i],"--filter")==0){understood=1;filter=1;};			// just filter
				if (strcmp(argv[i],"--p")==0){										// set precision 
				   precision=atof(argv[i+1]);
				   precision=pow(10.,(-precision));
				   numeric=1;
				   understood=1; };
				if (strcmp(argv[i],"--it")==0){										// maximum iteration
				   maxit = strtoul(argv[i+1],&pEnd,10);
				   numeric=1;
				   understood=1; };
				if (strcmp(argv[i],"--cov")==0){understood=1;cov=1; };				// print covariance matrix
				if (strcmp(argv[i],"--s")==0){understood=1;slow=1;};				// slow down adjustment
				if (strcmp(argv[i],"--slow")==0){understood=1;slow=1;};				// slow down adjustment

		
	 if (!understood) {printf("\nError: Cannot understand all command line switches.") ;
								printf("\nFor help, type vcc\n"); 
						 
				   return 1;}
	
			   }

		   };

 
	};
	
	zerovar=0;								// flag for zero variances in input
	negvar=0;								// flag for negative variance input
	
	
// Open input file

	    strcpy(infile,argv[1]);
		file = fopen(infile, "rt");
    if (!file) {
		printf("\nFile opening error\n");
		return 1;  
		}									// File opening error
    strcpy(outfile,infile);
	outfile[strlen(outfile)-4]='\0';		// truncate filename
	strcat(outfile,"-VCC.csv");				// add extension for output file name


// Read data from file. (Create matrix x and vector y, read in row name rname)
	
	n = 0;
	do {													// count columns in first row
		fscanf(file, "%[^,\n]%c ", name, &c);				// and read in column names
#ifdef LINUX
		if (c == '\n') name[strlen(name)-1]='\0';			// remove line feed for LINUX
#endif
		VECTOR(cname, n+1);
		cname[n] = 0; VECTOR(cname[n], strlen(name) + 1);
		strcpy(cname[n], name);
		++n;		
	} 
	
	while (c == ',');
	if (n < 3) return 1;
	
		n -= 2;	 								// n = width of x (first column is ignored, 
		                                        // second column is y)	
// VECTORS	 

	VECTOR(s, n);							// Generate variance vector s
    VECTOR(sini,n);							// Initial values of the variances
	VECTOR(rcalc,n);
	VECTOR(rfilter,n);						// Vector for variance ratios used in the last run
	VECTOR(rini,n);							// Create vector of initial variance ratios
	VECTOR(flag,n);
	for (i=0; i<n; i++) {flag[i]=0;};
		

// Take initial variances from input


	fscanf(file, "%*[^,],%lg ", &s0);	 	 // modifies original read s0
	
	
		for (i=0; i<n; i++) {
          
			fscanf(file, ",%lg ", &s[i]);
			rini[i]=s[i];					 //	read variance ratio for coefficient i
			
			// Read variances s 		
			if (s[i]==0) {
						s[i] = 1e-10;					// if zero variance ratio, make it positive and freeze it
						flag[i]=2;
					    zerovar=1;
				};	
			
			if (s[i]<0) {
						s[i] = -s[i];					// if negative variance ratio, make it positive and freeze it
					    flag[i]=1;
						negvar=1;
				};							
			
			sini[i]=s[i];									// Save initial variances
				
		

	}														// end of main calculations

	
		
		// Messages on variance input

	if (zerovar){
		strcpy(message,"\nSome initial variance ratios are zero. I have frozen");
		strcat(message,"\nthem close to zero to enforce constancy of the ");
		strcat(message,"\nassociated coefficient.\n");
		printf("%s",message);
	}
	
	
	if (negvar){
		strcpy(message, "\nSome initial variance ratios were negative. I have frozen");
		strcat(message,"\nthem to their absolute values.\n");
		printf("%s",message);
	}		
	
	
		T = 0;													// T = Lnge von x bzw. y

		
		while (!feof(file)) {										
			VECTOR(y, T+1), MATRIX(x, T+1, n);					// increase length of x and y
			VECTOR(rname, T+1); 								// increase length of rname 
			fscanf(file, "%[^,],%lg ", name, &y[T]);			// first column (name of rows) and value of y
			rname[T] = 0; VECTOR(rname[T], strlen(name) + 1);	// save name of columns
			strcpy(rname[T], name);
			for (i=0; i<n; i++) fscanf(file, ",%lg ", &x[T][i]);// get row of x
			T++;												// increase T by one
		}

	
	fclose(file);												// Close input file								

	// Generate vectors  r, z and matrix A
	i=T*n;

	// The vector of variances  (s(0), s(1), ..., s(N))" is here s0, s[0], ..., s[n-1].).
	VECTOR(r, n), VECTOR(z, n); MATRIX(A, T, n); MATRIX(AS, T, n); VECTOR(as, i);
	
	// Initial values of variance ratios r 

		for (i=0; i<n; i++) r[i] = s[i] ;



	// Solve normal equation system for A and 
	// determine iterated variances and variance ratios

	// This is the main program loop
	// =============================

	i = 0;
	if (!filter){	
	do {
	sol(x, y, A, r, z);		// Construct system matrix A for given observations x ynd y
							// and given variance ratios
							// Determine the vector z with elements z(i) = tr((P(i)*INV(CC')*P'(i))
							// solve normal equation, return address of C



		for (j=0; j<n; j++) 
			if 	(flag[j]) s[j]=sini[j];			// force selected variances to given values
		
		v = var(x, y, A, r, &s0, s, z);
	    

			i=i+1;
			printf("\rIteration : %u,   error: %-.6lf  per cent.", i, v*100);
		
	icalc=i;
		} while ((v >= precision) &&(i<maxit)&&(!filter)) ;
		}
	// LAST
	
	vcalc=v;
	for (j=0; j<n; j++) rcalc[j]=r[j];
	s0calc=s0;



		for (j=0; j<n; j++) {
			
			if (smooth !=0)	r[j]=r[j]/smooth;
			rfilter[j]=r[j];
		}
		sol(x, y, A, r, z);	
		v = var(x, y, A, r, &s0, s, z);
		   
	;

	// end main program loop

	if(i==maxit) {
		convergence=0;
		printf("\nRequired precision of %g per cent not reached after %u iterations.", precision*100, maxit);
		printf("\nIncrease number of iterations with the --it switch,");
		printf("\nIf results do not converge, try the --s switch.\n");
		}
	else{
		convergence=1;
		printf("\rAccuracy of %-.2lg  per cent. obtained after %u iterations.\n", vcalc*100,icalc);
	}

	
	standarderror(C, as);
	for (i=0;i<n*T;i++) as[i]=sqrt(s0*as[i]);


	// Save result to file and exit program
	// Write input and output to output file
	
	file = fopen(outfile, "wt");
	if (!file){ printf("\nError opening output file\n"); return 1;}			// file creation error

	fprintf(file, "%s,%s", cname[0], cname[1]);
	for (i=0; i<n; i++) fprintf(file, ",%s", cname[i+2]);
	for (i=0; i<n; i++) fprintf(file, ",a_%s,stderr_%s,lb_%s,ub_%s", cname[i+2],cname[i+2],cname[i+2],cname[i+2]);
	fprintf(file, "\n");
	
	for (t=0; t<T; t++) {
		fprintf(file, "%s,%lg", rname[t], y[t]);
		for (i=0; i<n; i++) fprintf(file, ",%lg", x[t][i]);
		for (i=0; i<n; i++) fprintf(file, ",%lg,%lg,%lg,%lg", A[t][i],as[t*n+i],A[t][i]-2*as[t*n+i],A[t][i]+2*as[t*n+i]);
		fprintf(file, "\n");
	}

	fprintf(file, "\n");
	if (filter) fprintf(file, "\n,No iteration. Time series just filtered.\n\n");   
	
	
	fprintf(file, ",,equation,,");
	for (i=0; i<n; i++) fprintf(file, "%s,,", cname[i+2]);
	fprintf(file, "\n");

	
	fprintf(file, "\n,Filtered with variance ratios");
	fprintf(file,",,,");
	for (i=0; i<n; i++) fprintf(file, "%lg,,", rfilter[i]);

	fprintf(file, "\n,Implied variances");
	fprintf(file,",%lg,,",s0filter);
	for (i=0; i<n; i++) fprintf(file, "%lg,,", rfilter[i] * s0filter);

	fprintf(file, "\n");

	if ((smooth !=0)&&(!filter)){
		fprintf(file, "\n,Estimated variance ratios,");
		fprintf(file, ",,");
		for (i=0; i<n; i++) fprintf(file, "%lg,,", rcalc[i]);

		fprintf(file, "\n,Implied variances,");
		fprintf(file, "%lg,,", s0calc);
		for (i=0; i<n; i++) fprintf(file, "%lg,,", rcalc[i] * s0calc);
		fprintf(file, "\n");
	};

	fprintf(file, "\n,Initial variance ratios");
	fprintf(file,",,,");
	for (i=0; i<n; i++) fprintf(file, "%lg,,", rini[i]);
	fprintf(file, "\n\n");
	
	if ((zerovar||negvar)&&(!filter)){
		
		for (i=0; i<n; i++) {
			if 	(flag[i]==1) {
				fprintf(file, ",Variance ratio of \"%s\" constant.\n", cname[i+2]);
				};
			if 	(flag[i]==2) {
				fprintf(file, ",Coefficient of \"%s\" constant.\n", cname[i+2]);
				}
			}
		}
	if (!filter) fprintf(file, ",%i iterations\n,Relative error %.2G per cent.\n", icalc, vcalc*100);

	if (vcalc>=.00001){				// Warning for lack of convergence
		fprintf(file,"\n\nWARNING! \nRequired precision of 0.001 per cent \nnot reached after %u iterations.", i);

	}; 
    
		fprintf(file,"\n\n,Output generated by the \n,VCC program by\n,Ekkehart Schlicht. Check");
		fprintf(file,"\n,\"epub.ub.uni-muenchen.de\"\n,for the most recent version.");	
	
	printf("\nOutput saved to %s\n",outfile);

	fclose(file);



	// Write covariance matrix

	if (cov) {
	
		unsigned int i, j, n, m, k; 
		n=LENGTH(C);								// rows
		m=LENGTH(*C);								// columns	
 		k=(n-1)*(m-1);

		outfile[strlen(outfile)-4]='\0';			// truncate filename
		strcat(outfile,"-covariances.csv");         // add extension for output file name
			
			file = fopen(outfile, "wt");
			if (!file) {
				printf("\nCould not generate output file for the covariance matrix.\n");
				return 1;	}						// File creation error
			
			for (i=0;i<n ;i++){
				for (j=0;j<n ;j++) as[j]=0;
				as[i]=1;
				errormatrix(C,as);
				for (j=0;j<(n-1) ;j++) fprintf(file, "%lg,",s0*as[j]);
				fprintf(file, "%lg\n",s0*as[n-1]);
			}

			fclose(file);
			printf("\nCovariance matrix saved to ");
			printf("%s",outfile);
			printf("\n");
	}	
	
	return 0;
}

// Finished


// Routines
// ========

// Program for solving the normal equation


//  x, y and r are given, the system matrix A is determined.

void sol(double **x, double *y, double **A, double *r, double *z)
{
	void cholband(double **A);
	void trace(double **C, double *z);
	void bandsol(double **A, double *x);

   		static double *a;	
		unsigned int T, n, t, i, j, q;
		double p;

	// Dimensions of  A  are T and n;
	T = LENGTH(x);
	n = LENGTH(*x);
	
	
	if (!C) MATRIX(C, T*n, n+1);

	for (j=1; j<T*n; j++) for (i=0; i<n+1; i++) C[j][i] = 0;

	for (t=0; t<T; t++) {
		p = y[t];
		q = t * n;
		for (i=0; i<n; i++) {
			A[t][i] = x[t][i] * p;
			for (j=0; j<=i; j++) C[q+i][n-i+j] = x[t][i] * x[t][j];
		}
	}
	
	for (i=0; i<n; i++) r[i] = 1/r[i];
	
	for (t=1; t<T; t++) {
		q = t * n;
		for (i=0; i<n; i++) C[q+i][0] = -r[i], C[q+i][n] += 2*r[i];
	}
	
	for (i=0; i<n; i++) C[i][n] += r[i], C[q+i][n] -= r[i], r[i] = 1/r[i];
	
	cholband(C);
	trace(C, z);

	// Redimension matrix A into vector a
	if (!a) VECTOR(a, T*n);
	{double *g; g = a; for (t=0; t<T; t++) for (i=0; i<n; i++) *g++ = A[t][i];}

	bandsol(C, a);
	
	// Undo redimensioning
	{double *g; g = a; for (t=0; t<T; t++) for (i=0; i<n; i++) A[t][i] = *g++;}

}




// Program for determining the standard deviations 
void standarderror(double **C, double *as)
{
	void bandsol(double **A, double *as);
	void bandtrn(double **A);
	void bandtrninv(double**A);
	unsigned int i, j, n, m; 
	double *e=0;

	n=LENGTH(C);													// rows
																	
	m=LENGTH(*C);													// columns
	
	
	VECTOR(e,n);

	
		for (i=0; i<n; i++) { 
			for (j=0; j<n; j++) e[j]=0; 
				e[i]=1;
				bandtrninv(C);	
				bandsol(C,e);
				as[i]=e[i];
		}


}

// Program for determining a row of the covariance matrix 
void errormatrix(double **C, double *as)
{
	void bandsol(double **A, double *as);
	void bandtrn(double **A);
	void bandtrninv(double**A);
		
		bandtrninv(C);	
		bandsol(C,as);
		
}

// A is overwritten with the transpose of A (stored in short format)
void bandtrn(double **A)
{
	unsigned int m, n, i, j;

	n = LENGTH(A);
	m = LENGTH(*A);
	
	for (i=0; i<n; i++) for (j=0; j<m && j<n-i; j++) A[i][j] = A[i+j][m-1-j];
	for (j=1; j<m; j++) for (i=n-j; i<n; i++) A[i][j]=0;
}

// Invert bandtrn, A is overwritten with the transpose of A (stored in short format)
void bandtrninv(double **A)


{
	unsigned int m, n, i, j;

	n = LENGTH(A);
	m = LENGTH(*A);
				
	for (i=0; i<n; i++) for (j=0; j<m && j<n-i; j++) A[n-1-i][m-1-j] = A[n-1-i-j][j];
	for (j=1; j<m; j++) for (i=n-j; i<n; i++) A[n-1-i][m-1-j]=0;
}

// bandsol1 solves A*y=x and overwrites x with y:
// A is lower triangular matrix briefly stored in short format
void bandsol1(double **A, double *x)
{
	unsigned int m, n, p, i, j;
	n = LENGTH(A);
	m = LENGTH(*A);
	p = m-1;
	
	for (i=0; i<n; i++) {
		for (j = i<p? p-i:0; j<p; j++) x[i] -= A[i][j] * x[j-(p-i)];
		x[i] /= A[i][p];
	}


}

// Like bandsol1, but for upper triangular matrix
void bandsol2(double **A, double *x)
{
	unsigned int m, n, i, j;

	n = LENGTH(A);
	m = LENGTH(*A);
	
	for (i=n; i>0;) {i--;
		for (j=1; j<m && j<n-i; j++) x[i] -= A[i][j] * x[i+j];
		x[i] /= A[i][0];
	}
}

// bandsol solves (A*A')*y=x and overwrites x with y.
// A is lower triangular matrix, stored in short format.
void bandsol(double **A, double *x)
{
	bandsol1(A, x);
	bandtrn(A);
	bandsol2(A, x);

}

// Cholesky decomposition of the band matrix A that is stored in short format.
// The input A is overwritten with lower triangular matrix A
void cholband(double **A)
{
	unsigned int n, m, p, i, j, k;
	
	n = LENGTH(A);
	m = LENGTH(*A);
	p = m-1;

	for (i=0; i<n; i++) {
		for (j = i>p? i-p:0; j<i; j++) {
			for (k = i>p? i-p:0; k<j; k++) A[i][p+j-i] -= A[i][p+k-i]*A[j][p+k-j];
			A[i][p+j-i] /= A[j][p];
		}
		for (k = i>p? i-p:0; k<i; k++) A[i][p] -= A[i][p+k-i] * A[i][p+k-i];
		A[i][p] = sqrt(A[i][p]);
	}
}

// trace determines z(i) = tr{P(i)*INV(CC')*P'(i)}
// for i = 1 to n, C is lower triangular matrix in short format.
void trace(double **C, double *z)
{
	unsigned int m, n, T, i, j, t;
	static double *x, *y; // (These vectors are created only once at the first function call).
	double q;
	
	m = LENGTH(C);
	n = LENGTH(z);
	T = m / n;
	
	if (!x) VECTOR(x, m);
	if (!y) VECTOR(y, m);
	
	for (i=0; i<n; i++) {
		q = 0;
		for (j=0; j<m; j++) y[j] = 0;
		y[i] = 1;
		bandsol1(C, y);
		for (t=1; t<T; t++) {
			for (j=0; j<m; j++) x[j] = y[j], y[j] = 0;
			y[t*n+i] = 1;
			bandsol1(C, y);
			for (j=0; j<m; j++) x[j] = y[j] - x[j], q += x[j] * x[j];
		}
		z[i] = q;
	}

}

// var determines new variance ratios and variances
double var(double **x, double *y, double **A, double *r, double *s0, double *s, double *z)
{
	static double *u, *r0; // (These vectors are created only once at the first function call).
	unsigned int T, n, t, i, con;
	double p, q, f;
	
	T = LENGTH(x);
	n = LENGTH(*x);
	con = T-n;         
	if (!u) VECTOR(u, T);
	if (!r0) VECTOR(r0, n);

	for (i=0; i<n; i++) r0[i] = r[i];

	for (t=0; t<T; t++) {
		u[t] = -y[t];
		for (i=0; i<n; i++) u[t] += A[t][i]*x[t][i];
	}

	q = 0; for (t=0; t<T; t++) q += u[t] * u[t];

	for (i=0; i<n; i++) {
		s[i] = 0;
		for (t=1; t<T; t++) f = A[t][i] - A[t-1][i], s[i] += f * f;
		q += s[i] / r[i];
	}
	*s0 = q / (T-n);
	s0filter=q / (T-n);

	p = 0;
	for (i=0; i<n; i++) {
		r[i] = (s[i] * con / q + z[i]) / (T-1);					// calculate new variance ratios
		if 	(!(flag[i])) {								// skip updating blocked variance ratios
				f = r[i] / r0[i];

		if (!slow) r[i] = r0[i] * pow(f, 20);					// fast relaxation, disabled by slow
		s[i] = r[i] * *s0;
		f -= 1; if (f > p) p = f; else if (-f > p) p = -f;
	}

	}
	return p;
}

