*-------------------------------------------------
* Authors:

* Christine Schnor
* Affiliation: Max Planck Institute for Demographic Research, Konrad-Zuse-Str.1, D-18057 Rostock. 
* Contact: schnor@demogr.mpg.de

* Sonja Bastin
* Affiliation: Max Planck Institute for Demographic Research, Konrad-Zuse-Str.1, D-18057 Rostock. 
* Contact: bastin@demogr.mpg.de


**************************************************************************************************
* The files Eventhistory.do and Eventhistory.dta may be used by other users by citing the pairfam 
* Technical Report "Partners and Babies" (Schnor/Bastin 2013). 
* However, we accept no responsibility for errors that may have arisen during the coding procedures.
**************************************************************************************************
  
*-------------------------------------------
* Wave 			: 1-3
* Stata do-file	: Eventhistory.do
* Release 3.1 (pairfam)
*		  2.0 (pairfam)
*		  2.0 (DemoDiff) 	
* Date of submission: 12/06/2013 
*---------------------------------------------------------------------------------------
* >> content		
* 	 PARTNERSHIP AND FERTILITY EVENT HISTORY DATA SET 
*
* 2 main steps:
* I) 	Prepare flag variables that mark imputed information
* II) 	Generate event history data set that combines the partnership and fertility biographies
*---------------------------------------------------------------------------------------
	
clear all
set mem 10g
set maxvar 10000
set more off
set varabbrev on


* Define your personal data paths
*************************************
global EHDIR "XXXXXX"


*************************************************************
* CHAPTER 1: PREPARATION OF THE BIOGRAPHIES
*************************************************************

		do "biopart_PF.do"
		do "biopart_DD.do"

*******************************************
* Section I: Flagging imputed information *
*******************************************

**** set path to directory where pairfam data are stored
	set more off 

	cd "$EHDIR"
	use anchor1.dta, clear				
	g DEMODIFF=0
	append using anchor1_DD.dta
	recode DEMODIFF .=1
	label language en
	

* define data set
	keep id pid  											/// 		/* ID anchor and current partner */
	sex_gen													///			/* anchor's sex (generated version) */
	psex_gen												/// 		/* sex current partner (generated version) */
	sd5e*em sd5e*ey sd5e*bm sd5e*by 						/// 		/* episodes current partner */
	rtr1p*g 												/// 		/* sex previous partners */
	rtr2p*e*bm rtr2p*e*by rtr2p*e*em rtr2p*e*ey rtr4p* 		/// 		/* episodes previous partners (incl. type of break) */
	sd7e1 sd7e2 sd8e*bm sd8e*by sd8e*em sd8e*ey 			/// 		/* cohabitation current partner */
	rtr7p*e*bm rtr7p*e*by rtr7p*e*em rtr7p*e*ey 			/// 		/* cohabitation previous partners */
	sd12m sd12y sd13 										/// 		/* marriage current partner */
	rtr9 rtr10* rtr11* 										/// 		/* current marriage previous partners */
	rtr13p* rtr14p*bm rtr14p*by rtr15p* rtr14p*em rtr14p*ey /// 		/* previous marriages previous partners */
	pa2m pa2y pa3 											/// 		/* meeting current partner */
	dobm_gen doby_gen pdobm_gen pdoby_gen					/// 		/* date of birth anchor and current partner (generated versions) */
	intd intm inty 											/// 		/* date of interview */
	flag5-flag8 flag10 flag11 flag12 flag14					///			/* flag variables  */
	homosex																/* generated variable "Anchor homosexual" */ 

	
* format id 
	format id %12.3g

* INTENTION: flag variables for (later) imputed months
		*note: some flag variables  has to be generated in two steps. 


***UNION***
*union start

	g IMP_begp0=0
	forvalues x=1/8 {
		capture noisily replace IMP_begp0=1 if (sd5e`x'bm==-1| sd5e`x'bm==-2) & sd5e`x'by>0
		capture noisily replace IMP_begp0=2 if sd5e`x'bm==21| sd5e`x'bm==24 | sd5e`x'bm==27| sd5e`x'bm==30 | sd5e`x'bm==32
		}
	lab var IMP_begp0 "Imputed missing month in (repeated) union start with current partner"


	forvalues x=1/12 {
	g IMP_begp`x'=0
	forvalues y=1/5 {
		capture noisily replace IMP_begp`x'=1 if (rtr2p`x'e`y'bm==-1| rtr2p`x'e1bm==-2)& rtr2p`x'e`y'by>0
		capture noisily replace IMP_begp`x'=2 if rtr2p`x'e`y'bm==21| rtr2p`x'e`y'bm==24 | rtr2p`x'e`y'bm==27| rtr2p`x'e`y'bm==30 | rtr2p`x'e`y'bm==32
	lab var IMP_begp`x' "Imputed missing month in in (repeated) union start with `x'. partner"
	}
	}

*union end 

	g IMP_endp0=0
	forvalues x=1/8 {
		capture noisily replace IMP_endp0=1 if (sd5e`x'em==-1| sd5e`x'em==-2) & sd5e`x'ey>0
		capture noisily replace IMP_endp0=2 if sd5e`x'em==21| sd5e`x'em==24 | sd5e`x'em==27| sd5e`x'em==30 | sd5e`x'em==32
		}
	lab var IMP_endp0 "Imputed missing month in (repeated) union end with current partner"
	
	

	
	forvalues x=1/12 {
	g IMP_endp`x'=0
	forvalues y=1/5 {
		capture noisily replace IMP_endp`x'=1 if (rtr2p`x'e`y'em==-1| rtr2p`x'e`y'em==-2) & rtr2p`x'e`y'ey>0
		capture noisily replace IMP_endp`x'=2 if rtr2p`x'e`y'em==21| rtr2p`x'e`y'em==24 | rtr2p`x'e`y'em==27| rtr2p`x'e`y'em==30 | rtr2p`x'e`y'em==32
	lab var IMP_endp`x' "Imputed missing month in union end with `x'. partner"
	}
	}

	
***COHABITATION***
*cohabitation start

	g IMP_cohp0beg=0
	forvalues x=1/5 {
		capture noisilyreplace IMP_cohp0beg=1 if (sd8e`x'bm==-1| sd8e`x'bm==-2) & sd8e`x'by>0
		capture noisily replace IMP_cohp0beg=2 if sd8e`x'bm==21| sd8e`x'bm==24 | sd8e`x'bm==27| sd8e`x'bm==30 | sd8e`x'bm==32
	lab var IMP_cohp0beg "Imputed missing month in cohabitation start with current partner"
	}
	
	forvalues x=1/12 {
	g IMP_cohp`x'beg=0
	forvalues y=1/5 {
		capture noisily replace IMP_cohp`x'beg=1 if (rtr7p`x'e`y'bm==-1| rtr7p`x'e`y'bm==-2) & rtr7p`x'e`y'by>0
		capture noisily replace IMP_cohp`x'beg=2 if rtr7p`x'e`y'bm==21| rtr7p`x'e`y'bm==24 | rtr7p`x'e`y'bm==27| rtr7p`x'e`y'bm==30 | rtr7p`x'e`y'bm==32
	lab var IMP_cohp`x'beg  "Imputed missing month in cohabitation start with `x'. partner"
	}
	}
	
*cohabitation end

	g IMP_cohp0end=0
	forvalues x=1/5 {
		capture noisily replace IMP_cohp0end=1 if (sd8e`x'em==-1| sd8e`x'em==-2) & sd8e`x'ey>0
		capture noisily replace IMP_cohp0end=2 if sd8e`x'em==21| sd8e`x'em==24 | sd8e`x'em==27| sd8e`x'em==30 | sd8e`x'em==32
	lab var IMP_cohp0end "Imputed missing month in cohabitation end with current partner"
	}
	
	forvalues x=1/12 {
	g IMP_cohp`x'end=0
	forvalues y=1/5 {
		capture noisily replace IMP_cohp`x'end=1 if (rtr7p`x'e`y'em==-1| rtr7p`x'e`y'em==-2) & rtr7p`x'e`y'ey>0
		capture noisily replace IMP_cohp`x'end=2 if rtr7p`x'e`y'em==21| rtr7p`x'e`y'em==24 | rtr7p`x'e`y'em==27| rtr7p`x'e`y'em==30 | rtr7p`x'e`y'em==32
	lab var IMP_cohp`x'end "Imputed missing month in `y'. cohabitation end with `x'. partner"
	}
	}
	
***MARRIAGE***
*marriage start

	g IMP_marp0beg=0
		replace IMP_marp0beg=1 if (sd12m==-1| sd12m==-2) & sd12y>0
		replace IMP_marp0beg=2 if sd12m==21| sd12m==24 | sd12m==27| sd12m==30 | sd12m==32
	lab var IMP_marp0beg "Imputed missing month in marriage start with current partner"
	
	forvalues x=1/9 {
	g IMP_marp`x'beg=0
		capture noisily replace IMP_marp`x'beg=1 if (rtr14p`x'bm==-1| rtr14p`x'bm==-2) & rtr14p`x'by>0
		capture noisily replace IMP_marp`x'beg=2 if rtr14p`x'bm==21| rtr14p`x'bm==24 | rtr14p`x'bm==27| rtr14p`x'bm==30 | rtr14p`x'bm==32
	lab var IMP_marp`x'beg "Imputed missing month in marriage start with `x'. partner"
	}
	
*marriage end
	g IMP_marp0end=0

	forvalues x=1/9 {
	g IMP_marp`x'end=0
		capture noisily replace IMP_marp`x'end=1 if (rtr14p`x'em==-1| rtr14p`x'em==-2) & rtr14p`x'ey>0
		capture noisily replace IMP_marp`x'end=2 if rtr14p`x'em==21| rtr14p`x'em==24 | rtr14p`x'em==27| rtr14p`x'em==30 | rtr14p`x'em==32
	lab var IMP_marp`x'end "Imputed missing month in marriage end with `x'. partner"
	}	
	
	keep id IMP* 
	sort id
	cd "$EHDIR"
	save _IMP.dta, replace
	

* load biopart from wave 1, version 2.0 (interim data set of biopart.do)
	set more off
	
	cd "$EHDIR"
	use biopart1_v2.dta, clear

	
	g DEMODIFF=0
	append using _biopart1_v2_DD.dta
	recode DEMODIFF .=1 
	sort id

	merge 1:1 id using _IMP.dta
	drop _merge



	foreach var of varlist IMP* {
		lab def `var' 1 "Only year information" 2 "Only season information"
		lab value `var' `var'
		}

	foreach var of varlist IMP* {
				replace `var'=0 if `var'==. 
			}
	/*
foreach var of varlist IMP* {
			count if `var' ==0
			if r(N) == _N drop `var'
		}
*/


************************************
* reshape this data to long format * 
************************************

	reshape clear
	reshape i id
	reshape j partner
	reshape xij dobp@ meetp@ waymeetp@ sexp@ begp@ endp@ b1p@beg b2p@beg b3p@beg b4p@beg ///
		b5p@beg b6p@beg b7p@beg b1p@end b2p@end b3p@end b4p@end b5p@end b6p@end b7p@end ///
		reasendp@ cohp@beg b1cohp@beg b2cohp@beg b3cohp@beg b1cohp@end b2cohp@end b3cohp@end ///
		cohp@end marp@beg marp@end marp@cer dropp@ indexp@ ///
		IMP_begp@ IMP_endp@ IMP_cohp@beg IMP_cohp@end IMP_marp@beg IMP_marp@end 
	reshape long


*******************************	
**** modifications / corrections data wave 1
*******************************	

* reasendp current partner: .=-3 (did not exist in biopart wave 1)
	recode reasendp .=-3

* format
	format %12.0f id

* droppx: -3 if partner does not exist
	** will be solved when deleting empty variables

* delete cases with sexp =-2 or -1 and everything else -3
	count if sexp==-1
	count if sexp==-2
	*list sexp begp endp cohpbeg cohpend marpbeg marpend flag* if sexp==-2 | sexp==-1
* If partner's sex is "-1" or "-2", no additional questions regarding this partner (see filter) 
* Assumption: Partner does not exist --> -3
	gen help=0
	foreach var of varlist begp-marpend {
		bysort id: replace help=help+1 if `var'==-3
	}
	tab sexp, m
	l id if (sexp==-2 | sexp==-1) & help==7
	replace sexp=-3 if (sexp==-2 | sexp==-1) & help==7
	drop help
	
* drop partners who have been identified as inconsistencies in wave 1 (see variable "dropp")
	tab dropp, m
	drop if dropp==1
	drop dropp

* delete empty variables 
	gen help=0
	foreach var of varlist sexp-marpend {
		bysort id: replace help=help+1 if `var'==-3
	}
	tab help, m
	drop if help==8
	drop help

* missing values: -3 instead of .
	foreach var of varlist dobp meetp waymeetp b1pbeg-b3cohpend {
		recode `var' .=-3
	}	
	

	foreach var of varlist IMP* {
			replace `var'=0 if `var'==. 
		}
		
* pid: only if current partner, else -3 	
	replace pid=-3 if partner!=0
	
* marpcer: 4 cases without information regarding marriage ceremony despite of marriage 	
	l if marpbeg!=-3 & marpcer==-3
	* reason: wrong partner number; see data set "anchor1"
	replace marpcer=1 if id==133823000 & indexp==3
	replace marpcer=-2 if id==138256000 & indexp==3
	replace marpcer=-2 if id==185131000 & indexp==3
	replace marpcer=3 if id==207387000 & indexp==1 // partner 1 has been dropped due to consistency checks
	l if id==133823000 | id==138256000 | id==185131000 | id==207387000

	drop if begp==. & DEMODIFF==1 & partner>9
	
*******************************	
**** generate new variables, data wave 1
*******************************	

* current partner wave 1: currw1
	gen currw1=0
	replace currw1=1 if partner==0
	tab currw1 partner, m

* rename "partner" --> "pnow1"
	rename partner pnow1
	
* dodp (see reasendp)
	tab reasendp, m
	l if reasendp==3
	gen dodp=-3
	egen end=rowmax(endp cohpend marpend) if reasendp==3
	tab end, m
	l if end<0
	recode end -3=-88
	l if end<0
	replace dodp=end if reasendp==3
	count if end==endp & reasendp==3 
	count if end==cohpend & reasendp==3
	count if end==marpend & reasendp==3
	foreach var of varlist endp cohpend marpend {
		replace `var'=-66 if reasendp==3 & end==`var'
	}
	l if reasendp==3
	drop end
	drop reasendp
	

*******************************	
**** modifications of variables, data wave 1
*******************************	

* no "p" as part of variable labels (exceptions: sexp and dobp to distinguish from anchor)
	foreach var in beg end {
		rename `var'p `var'
		rename cohp`var' coh`var'
		rename marp`var' mar`var'
		rename IMP_`var'p IMP_`var'
		rename IMP_cohp`var' IMP_coh`var'
		rename IMP_marp`var' IMP_mar`var'
		foreach num of numlist 1/7 {
			capture noisily rename b`num'p`var' b`num'`var'
			capture noisily rename b`num'cohp`var' b`num'coh`var'
		}
	}		
	rename marpcer marcer
	rename indexp index
	rename intdat intdatw1
	
* delete variables which won't be included in "biopart" any more
	drop meetp
 	drop waymeetp
	drop flag5-flag14

* order of variables
	order id sex dob intdatw1 index pid pnow1 sexp dobp dodp beg end b1beg-b7end cohbeg cohend b1cohbeg-b3cohend ///
	marbeg marend marcer currw1  IMP*

* label variables when label is missing
	lab var intdatw1 "Date of interview wave 1"
	lab var index "Correct order of relationships: Number partner"
	lab var pnow1 "Number partner in wave 1"
	lab var sexp "Sex partner"
	lab var dobp "Date of birth partner"
	lab var dodp "Date of death partner"
	lab var beg "Beginning relationship"
	lab var end "End relationship"
	lab var cohbeg "Beginning cohabitation"
	lab var cohend "End cohabitation"
	lab var marbeg "Beginning marriage"
	lab var marend "End marriage"
	lab var marcer "Marriage ceremony"
	foreach num of numlist 1/7 {
		capture noisily lab var b`num'beg "Beginning break `num' relationship"
		capture noisily lab var b`num'end "End break `num' relationship"
		capture noisily lab var b`num'cohbeg "Beginning break `num' cohabitation"
		capture noisily lab var b`num'cohend "End break `num' cohabitation"
	}
	lab var currw1 "Current partner in wave 1"

* value labels when label is missing
	label list
	lab def dobp0 -66 "-66 Death partner", modify
	lab def date1 -66 "-66 Death partner", modify
	lab val dobp dobp0
	lab val pid date2
	lab val dodp end cohend marend b*beg b*end date1
	lab val currw1 yesno

* sort
	sort id index
	by id: l id if beg[_n+1]<beg & beg[_n+1]>0

	
* any partners who are not p0 and end==-99? 
	l pnow1 beg end if end==-99 & pnow1!=0
		* no such cases

	
********************************************
**** save reshaped version of biopart wave 1
********************************************	
	compress
	datasignature set, reset
	
* version for pairfam data to add IMP variables to biopart data
	cd "$EHDIR"
	save _biopart1_v2_reshapedPDD.dta , replace
	
	use _biopart1_v2_reshapedPDD.dta , replace
	keep if DEMODIFF==0
	save biopart1_v2_reshaped.dta, replace
	
	
	use _biopart1_v2_reshapedPDD.dta , replace
	keep if DEMODIFF==1
	save _biopart1_v2_reshaped_DD.dta, replace

	
********************************************
* Now run biopartIMP.do 
*****************************************
	cd "$EHDIR"
	
	do "biopart_PF_IMP.do"
	
	use biopart.dta, clear
	sort id index
	keep id index IMP*
	save _IMP_P.dta, replace
	
	use biopart_R3.dta
	sort id index
	merge 1:1 id index using _IMP_P.dta
	

	foreach var of varlist IMP* {
			replace `var'=0 if `var'==. 
		}

	
	save _biopart.dta, replace
	*Final biopart-pairfam-version with IMP variables: biopart.dta
	
	*cd "$EHDIR"
	do "biopart_DD_IMP.do"
	*Final biopart-DemoDiff-version with IMP Variablen: biopart_DD.dta
	
	*>> now, the flag variables (IMP_$) are included in the biopart.dta

*********************************************************************************
* FINALYZE IMPUTED VARIABLES
*********************************************************************
	clear all
	set mem 500m
	set more off
	
	cd "$EHDIR"
	use _biopart.dta, clear
	g DEMODIFF=0
	append using biopart_DD.dta
	recode DEMODIFF .=1 

	*recode missings in DemoDiff data to -3
	foreach var of varlist b4coh* {
			replace `var'=-3 if `var'==. 
		}

* CHECK: ARE IMP_Variables correctly specified?
*****************************************************	
*control if relationship episodes were deleted because of inconsistencies

	l id if IMP_beg>0 & beg==-3		
	l id if IMP_end>0 & end==-3	
*no cases

* ONLY FLAG INCONSISTENCIES IF THEY WERE NOT DRIVEN BY RANDOM IMPUTATION!
* >> check  biopartflag1-4

*biopartflag1: marriage date before union start date

	l id beg marbeg  IMP_beg IMP_marbeg if  biopartflag1==1

* imputed variables are recoded
	replace beg=marbeg if IMP_beg!=0 & IMP_marbeg==0 & biopartflag1==1
	replace marbeg=beg if IMP_marbeg!=0 & biopartflag1==1

	replace biopartflag1=0 if (IMP_beg!=0 | IMP_marbeg!=0) & biopartflag1==1

*biopartflag2: indicates overlaps in cohabitation episodes
* is dropped because our own variable suits better for analysis 
* (will be re-generated within the event history syntax)
	drop biopartflag2
	
*biopartflag3: Beginning current and end previous marriage
	l id marbeg marend IMP_marbeg IMP_marend if biopartflag3==1

* no case where the imputation caused an overlapping marriage episode

*********************************************************
* Save final version of biopart pairfam AND DemoDiff

	cd "$EHDIR"
	save _biopart_PDD.dta, replace
	



************************************************************************
* Section II:  generation of cohabitation and marriage index variables *
************************************************************************

	clear all
	set mem 500m
	set more off


*COHABITATION INDEX
**************************************
* remarks: the biopart data contains only the partnership index. The following syntax generates an index for cohabitation partners.

	cd "$EHDIR"
	use _biopart_PDD.dta, clear

*before imputing: take care of the one missmatched case (id 204759000) 
* >>(cohabitation episode of 5th partner within partnership episode of 3th partner.) 
* solution: recode 3th episode and set 5th episode to missing.

	replace cohbeg=cohbeg[_n+2] if index==3 & id==204759000
	replace cohend=cohend[_n+2] if index==3 & id==204759000
	replace cohbeg=-7 if index==5 & id==204759000
	replace cohend=-7 if index==5 & id==204759000

*STRATEGY: 	  We impute the start of cohabitation as start of union formation, in case it is not missing.
			* Otherwise we recode it to the cohabitation end date, in case it is not missing.
			* If union and cohabitation dates are missing, we replace the missing value by a dummy (index), in case it is not preceded by cohabitation episodes with dates.
			
	sort id
	
	by id: replace cohbeg=beg if beg>0 & cohbeg==-7
	by id: replace cohbeg=cohend if beg<0 & cohend>0 & cohbeg==-7
	by id: replace cohbeg=index if cohbeg==-7 & beg==-7 & cohend==-7
	by id: replace cohbeg=cohbeg[_n-1]+index if beg<0 & (cohbeg>0 & cohbeg<=13) & cohbeg[_n-1]>0
	by id: replace cohbeg=cohbeg[_n-2]+index if beg<0 & (cohbeg>0 & cohbeg<=13) & cohbeg[_n-2]>0
	
	mvdecode _all, mv(-3=.a )

	keep id  cohbeg  index cohbeg cohend biopartflag* IMP*  b*cohbeg b*cohend
	sort id cohbeg 
	by id: g index_c=_n
	
	replace index_c=.a if cohbeg==.a
		
	lab var index_c 	"Index partner with whom the respondent lives together"
	lab def index_c  	0 "No domestic partner" ///
						1 "1st domestic partner" ///
						2 "2nd domestic partner" ///
						3 "3rd domestic partner" ///
						4 "4th domestic partner" ///
						5 "5th domestic partner" ///
						6 "6th domestic partner" ///
						7 "7th domestic partner"
	lab value index_c index_c
		
	keep id index_c index
	
	cd "$EHDIR"
	save _COHAB_LONG.dta, replace


*MARRIAGE INDEX
**************************************
	cd "$EHDIR"
	use _biopart_PDD.dta, clear

*recode missing information to keep the right marriage order
* STRATEGY: We impute the missing marriage start date with the union start date, if the union start date is not missing.
			* If the union date is missing too, we replace the missing value by a dummy (index)

	by id,sort: replace marbeg=beg if beg>0 & marbeg==-7
	by id, sort: replace marbeg=index if marbeg==-7 & beg==-7

    mvdecode _all, mv(-3=.a )
		
	keep id  marbeg  index 
	sort id marbeg 
	by id: g index_m=_n
	
	replace index_m=.a if marbeg==.a
		
	lab var index_m "Index married partner"
	lab def index_m 0 "No spouse" 1 "1st spouse" 2 "2nd spouse" 3 "3rd spouse" 4 "4th spouse"
	lab value index_m index_m
		
	keep id index_m index
 
	cd "$EHDIR"
	save _MARR_LONG.dta, replace


*MERGE COHABITATION AND MARRIAGE INDEX WITH BIOPART.DTA
****************************************************************

	cd "$EHDIR"
	use _biopart_PDD.dta, clear
	drop _merge
	sort id index
	merge 1:1 id index using _COHAB_LONG.dta
	drop _merge

	sort id index
	merge 1:1 id index using _MARR_LONG.dta
	drop _merge



	cd "$EHDIR"
	save "_biopart2_index.dta", replace

	
*********************************************************
* CHAPTER 2: PREPARATION OF BIOCHILD
********************************************************

**********************************************************
* Section I: Including childless persons
**********************************************************

* BEFORE DEALING WITH FERTILITY HISTORIES EXTRACT VARIABLES NEEDED LATER FROM ANCHOR1.dta and ANCHOR2.dta 
* AS OTHERWISE THEY WILL ONLY BE AVAILABLE FOR PEOPLE WITH CHILDREN OR PARTNERS, 
* BUT WE WANT TO INCLUDE ALWAYS CHILD- AND PARTNERLESS PEOPLE, TOO.

set more off
	* Merge with dob, intdat and several id variables from anchor1 +2 
	* Information from wave 1:
	cd "$EHDIR"
	use "anchor1.dta", clear
	g DEMODIFF=0
	append using "anchor1_DD.dta"
	recode DEMODIFF . =1
	
	label language en
	
* generate combined date of birth (anchor1)
	tab1 doby_gen dobm_gen, m
	* For DEMODIFF there is only doby and dobm (no x_gen) 
	tab1 doby dobm, m
	gen dob=ym(doby_gen, dobm_gen)
	replace dob=ym(doby, dobm) if DEMODIFF==1
	lab var dob "Date of birth anchor" 
	tab dob, m
	replace dob=dob-ym(1900,1) if dob>0

* generate combined date of 1st interview (anchor1)	
	gen intdatw1=ym(inty, intm)
	replace intdatw1=intdatw1-ym(1900,1)
	lab var intdatw1 "Date of 1st interview"
	
	rename pid PIDI
	
	keep id dob intdatw1 PIDI sex DEMODIFF
	sort id
	
	cd "$EHDIR"
	save _ids_dates1_DEMO.dta, replace 

* information from wave 2:	
	use "anchor2", clear
	append using "anchor2_DD"
	label language en
	
* generate combined date of 2nd interview (anchor2)
	gen intdatw2=ym(inty, intm)
	lab var intdatw2 "Date of 2nd interview"
	replace intdatw2=intdatw2-ym(1900,1)
	
	rename pid PIDII
	rename mid MID
	rename fid FID
	rename smid SMID
	rename sfid SFID
	rename cid CID
	
				
	keep id CID MID FID SMID SFID PIDII intdatw2
	sort id
	cd "$EHDIR"
	save _ids_dates2.dta, replace 

	
	sort id
	merge 1:1 id using "_ids_dates1_DEMO.dta"
	drop _merge
	
	cd "$EHDIR"
	save _ids_dates1_2DEMO.dta, replace

* information from wave 3:	
	use anchor3.dta,clear
	gen intdatw3=ym(inty, intm)
	lab var intdatw3 "Date of 3rd interview"
	replace intdatw3=intdatw3-ym(1900,1)
	keep id intdatw3
	sort id
	merge id using _ids_dates1_2DEMO.dta, sort
	drop _merge
	save _ids_dates1_2_3P_DD.dta, replace
	
	
* merge dates und ids with biochild	
	cd "$EHDIR"
	use  "biochild.dta", clear
	*g DEMODIFF=0
	append using "biochild_DD.dta"
	*recode DEMODIFF .=1

	sort id
	merge m:1 id using "_ids_dates1_2_3P_DD.dta"
	drop _merge

* drop further here not needed variables
	drop mid fid smid sfid cid pid


* rename labels for more simple data handling
	recode _all (.=.a)
	lab def sexk 		1"Male" 2"Female" -7"-7 Incomplete information" 
	lab def livkbirth 	0"Not living together since birth" 1"Living together since birth" -3"-3 Does not apply" -7"Incomplete information"
	lab def index 		-3 "-3 Does not apply" 1 "1st child" 2 "2nd child" 3 "3rd child" 4 "4th child" 5 "5th child" ///
						6 "6th child" 7 "7th child" 8 "8th child" 9 "9th child" 10 "10th child"

	lab val sexk sexk
	lab val livkbirth livkbirth
	lab val index index
	
	label drop sex livbirth indexk



*************************************
* Section II: Change data structure from long-long to long-wide
*************************************

* currently there is one line for each wave and each child.
* We need only one line for each child:

* 1. generate separate variables for each wave

	sort id index wave 

	foreach X of varlist intdat-dodk {
		g `X'I=`X' if wave==1
		g `X'II=`X' if wave==2
		g `X'III=`X' if wave==3
		lab val `X'I* `X'
		
		by id, sort: replace `X'II=`X'II[_n+1] if `X'II==. & index==index[_n+1]
		by id, sort: replace `X'III=`X'III[_n+1] if `X'III==.  & index==index[_n+1]
		by id, sort: replace `X'III=`X'III[_n+1] if `X'III==.  & index==index[_n+1]

		}

sort id index wave 
* drop the now redundant rows 
	by id: drop if waveI==. & index==index[_n-1]

*drop the now redundant variables
	foreach X of varlist wave-dodk {
		drop `X'
		}

* Summarize time constant variables: Only one information needed for each child 
* This might have been surveyed in wave 1 or in wave 2 or in wave 3. 
******************************
	foreach X of newlist dobk statusk sexk parentid index number   {
		g `X'=`X'I if `X'I!=.
		replace `X'=`X'II if `X'II!=.
		replace `X'=`X'III if `X'III!=.
		lab val `X' `X'
		}

 

*Time varying variables
	g surveykid=0
	replace surveykid = 1 if surveykidI==1 | surveykidII==1 | surveykidIII==1

* Drop redundant variables
******************************************
	drop dobkI*  statuskI*  sexkI*  parentidI* ///
	numberI* indexI*  surveykidI* wave*
	
	
* Some information are only being surveyed in wave 1. In these cases the variable for wave 2 and 3 can be deleted. 
* The variables keep the suffix "I" to show that the information comes from wave 1 and might be outdated now.
	drop livkbegII livkendII b1livkbegII b1livkendII livkbirthII
	drop livkbegIII livkendIII b1livkbegIII b1livkendIII livkbirthIII

	sort id index
	by id: gen index_n=_n

	cd "$EHDIR"
	save  "_bioch_long_wide.dta", replace


******************************************************
* Section III: Integrate flag variables in biochild data
******************************************************
* first step: generate flag variables for biochild
**************************************************

	clear all
	set mem 500m
	set more off

	cd "$EHDIR"
	use "anchor1", clear	
	g DEMODIFF=0
	append using "anchor1_DD"
	recode DEMODIFF .=1
	
* special preparation for modified do-file
	*label language en		// change to english labels
	label language de		// change to german labels
	format id %12.0g		// change to readable (concerning digits) format

	
* define data set
	keep id 	   				/// 		/* ID anchor */
	sd19k*m sd19k*y				/// 		/* date of birth children */
	rtr18k*m rtr18k*y  			///			/* date of death children */
	sd15k*						///			/* status of children */	
	sd16k*						/// 		/* second biological parent */
	rtr17k*e*bm rtr17k*e*by		/// 		/* beginning living together with children */
	rtr17k*e*em rtr17k*e*ey		/// 		/* end living together with children */
	rtr20k*m rtr20k*y 			///			/* right-censored : beginning living together with child (month year)*/
	intm inty					///			/* date of interview */
	dobm doby					///			/* date of birth anchor */
	flag1 flag15				///			/* relevant flag variables */
	k*sex_gen k*dob*_gen		///			/* gender & generated birth date info */
	DEMODIFF
	
	*flag later imputed variables
*********************************************
ds *m	k*dobm_gen			
	foreach var in `r(varlist)' {
		local ding:value label `var'
		if inlist("`ding'","liste10neu","liste22","liste23", "h_pass")  { 
			g IMP_`var'=1 if `var'==21 
			*lab copy var IMP_`var' `var'
			replace IMP_`var'=1 if `var'==21
			replace IMP_`var'=1 if `var'==24
			replace IMP_`var'=1 if `var'==27
			replace IMP_`var'=1 if `var'==30
			replace IMP_`var'=1 if `var'==32
			replace IMP_`var'=2 if `var'==-2
			replace IMP_`var'=2 if `var'==-1
		}
	}
	
	foreach var of varlist IMP* {
			count if `var' ==.
			if r(N) == _N drop `var'
		}

	* IMP_dobbiok: imputed month in birth date of biological child
	* IMP_dodbiok: imputed month in death date of biological child
	forvalues x=1/10 {
		cap rename IMP_sd19k`x' IMP_dobbiok`x'
		cap rename IMP_rtr18k`x'm IMP_dodbiok`x'
		cap recode IMP_dobbiok`x' .=0
		cap recode IMP_dodbiok`x' .=0
		cap replace IMP_dobbiok`x'=0 if sd15k`x'!=1
		cap replace IMP_dodbiok`x'=0 if sd15k`x'!=1
		}
	
	* IMP_livbiok`x'beg: imputed month in start of coresidence with biological child
	* IMP_livbiok`x'end: imputed month in end of coresidence with biological child
	* IMP_livnonbiok`x'beg: imputed month in start of coresidence with nonbiological child
	* IMP_livnonbiok`x'end: imputed month in end of coresidence with nonbiological child

	forvalues x=1/10 {
		g IMP_livbiok`x'beg=0
		lab var IMP_livbiok`x'beg "Imputed month in start date of coresidence with `x'. kid"
		cap replace IMP_livbiok`x'beg=1 if (IMP_rtr17k`x'e1bm==1 | IMP_rtr20k`x'm==1) & sd15k`x'==1
		cap replace IMP_livbiok`x'beg=2 if (IMP_rtr17k`x'e1bm==2 | IMP_rtr20k`x'm==2) & sd15k`x'==1
		g IMP_livbiok`x'end=0
		lab var IMP_livbiok`x'end "Imputed month in end date of coresidence with `x'. kid"
		cap replace IMP_livbiok`x'end=1 if IMP_rtr17k`x'e1em==1 & sd15k`x'==1
		cap replace IMP_livbiok`x'end=2 if IMP_rtr17k`x'e1em==2 & sd15k`x'==1
		g IMP_livnonbiok`x'beg=0
		lab var IMP_livnonbiok`x'beg "Imputed month in start date of coresidence with `x'. kid"
		cap replace IMP_livnonbiok`x'beg=1 if (IMP_rtr17k`x'e1bm==1 | IMP_rtr20k`x'm==1) & sd15k`x'!=1
		cap replace IMP_livnonbiok`x'beg=2 if (IMP_rtr17k`x'e1bm==2 | IMP_rtr20k`x'm==2) & sd15k`x'!=1
		g IMP_livnonbiok`x'end=0
		lab var IMP_livnonbiok`x'end "Imputed month in end date of coresidence with `x'. kid"
		cap replace IMP_livnonbiok`x'end=1 if IMP_rtr17k`x'e1em==1 & sd15k`x'!=1
		cap replace IMP_livnonbiok`x'end=2 if IMP_rtr17k`x'e1em==2 & sd15k`x'!=1
		}
		
		drop IMP_rtr17k* IMP_rtr20k* 
		
		
		foreach var of varlist IMP_livbiok* IMP_livnonbiok* {
			count if `var' ==0
			if r(N) == _N drop `var'
		}
		
		
		keep id IMP* sd19k*y DEMODIFF
		
		reshape clear
		reshape i id
		reshape j episode
		reshape xij  IMP_dobbiok@ IMP_dodbiok@ IMP_livbiok@beg IMP_livbiok@end IMP_livnonbiok@beg IMP_livnonbiok@end sd19k@y
		reshape long 
		rename episode number
		sort id number
		drop if  sd19ky==-3
		drop  sd19ky
		
	
		
		cd "$EHDIR"
		save "_IMP_biochild.dta", replace
		
		****************
		*second step: generate a matching variable
		
		cd "$EHDIR"
		use "biochild.dta", clear
		g DEMODIFF=0
		append using "biochild_DD.dta"
		recode DEMODIFF .=1
		*use "bioch_long_wide.dta", clear
		*! data contains also childless people!
		drop if wave!=1
		keep id index number DEMODIFF
		sort id number
		
		cd "$EHDIR"
		merge 1:m id number using "_IMP_biochild.dta"
		*merge id number using "\IMP_biochild.dta", sort
		drop if index==.
		drop _merge
		
		sort id index 
		by id: g index_n=_n
		keep id index_n IMP*
		
		sort id index_n
		merge 1:1 id index_n using "_bioch_long_wide.dta"
		
		drop index_n _merge
		
		*rename variables for consistent labeling
		rename IMP_livbiokbeg IMP_beglivbiok
		rename IMP_livbiokend IMP_endlivbiok
		rename IMP_livnonbiokbeg IMP_beglivnonbiok
		rename IMP_livnonbiokend IMP_endlivnonbiok


			foreach var of varlist  IMP_dobbiok IMP_dodbiok IMP_beglivbiok IMP_endlivbiok IMP_beglivnonbiok IMP_endlivnonbiok {
			replace `var'=0 if `var'==.
			
		}
		
		cd "$EHDIR"
		save "_index_n.dta", replace
	

	
********************************************
* CHAPTER 3: SPECIFIC RECODING AND TRANSFORMATION PROCEDURE 
********************************************

*******************************************************
* 3.1. Partnership biography
*******************************************************

	clear all
	set mem 500m
	set more off
	
	cd "$EHDIR"
	use _biopart2_index.dta, clear 

********************************************************
* Section I: Recode interview errors in episodes
**********************************************************

* Probable interviewer error: Start of break = End of union (End of break is missing in those cases)
* --> The whole break needs to be set on -3 (does not apply)

		local i=0
			while `i'<7  {
			local default1=`i'+1
				capture noisily {
				replace b`default1'beg=-3 if b`default1'beg==end & end!=-7
				replace b`default1'end=-3 if b`default1'beg==-3 & end!=-7
				
				replace b`default1'beg=-3 if b`default1'end==beg & beg!=-7
				replace b`default1'end=-3 if b`default1'beg==-3 & beg!=-7
				}
			local i=`i'+1
			}

* Probable interviewer error: Start of break = End of cohab (End of break is missing in those cases)
* --> The whole break needs to be set on -3 (does not apply)

		local i=0
			while `i'<7  {
			local default1=`i'+1
				capture noisily {
				replace b`default1'cohbeg=-3 if b`default1'cohbeg==cohend & cohend!=-7
				replace b`default1'cohend=-3 if b`default1'cohbeg==-3 & cohend!=-7
				*-> 2 cases
				replace b`default1'cohbeg=-3 if b`default1'cohend==cohbeg & cohbeg!=-7
				replace b`default1'cohend=-3 if b`default1'cohend==-3 & cohbeg!=-7
				*-> 0 cases
				}
			local i=`i'+1
			}

********************************************************
* Section II: Recode and flag episode as missing in case either the start or the end date is missing
**********************************************************

* recode union episode as missing in case either the start or the end date is missing.			
	replace beg=-7 if  end==-7 
	replace end=-7 if beg==-7 

*flag union in which either the beginning or the end of a union or a union break is missing 
			
	g FLAG_M_UNION=.
		replace FLAG_M_UNION=1 if beg==-7 | b1beg==-7 | b1end==-7
		lab def FLAG_M_UNION 1 "Union with missing dates"
		lab val FLAG_M_UNION FLAG_M_UNION
		lab var FLAG_M_UNION "FLAG: Union with missing dates"

* recode cohab episode as missing in case either the start or the end date is missing.			
	replace cohbeg=-7 if  cohend==-7 
	replace cohend=-7 if cohbeg==-7 

*flag cohab in which either the beginning or the end of a cohab or a cohab break is missing 
			
	g FLAG_M_COHAB=.
		replace FLAG_M_COHAB=1 if cohbeg==-7 | b1cohbeg==-7 | b1cohend==-7
		lab def FLAG_M_COHAB 1 "Cohab with missing dates"
		lab val FLAG_M_COHAB FLAG_M_COHAB
		lab var FLAG_M_COHAB "FLAG: Cohab with missing dates"
			
* recode marr episode as missing in case either the start or the end date is missing.			
	replace marbeg=-7 if marend==-7 
	replace marend=-7 if marbeg==-7 

*flag marr in which either the beginning or the end of marr is missing 
	g FLAG_M_MARR=.
		replace FLAG_M_MARR=1 if marbeg==-7 
		lab def FLAG_M_MARR 1 "Marriage with missing dates"
		lab val FLAG_M_MARR FLAG_M_MARR
		lab var FLAG_M_MARR "FLAG: Marriage with missing dates"

		
		
		
*****************************************************************************
* Section III: Homosexuality: generate new variable indicating same-sex unions 					
******************************************************************************
*the variable <homosex>, available in anchor.dta does not refer to previous homosexual partnership.
* In <homosex>, respondents are recoded as homosexual if the have a samesex relationship or they have a preference to search for a samesex partner.
* >> this variable is not appropriate for the event history approach; 
* therefore a new variable indicating the sexual orientation is later generated and the pairfam variable <homosex> is dropped here.

	drop homosex

	
		
	cd "$EHDIR"
	save "_biopart_flagged.dta", replace

	
	
	
*****************************************************************************	
*****************************************************************************	
* 3.2. Fertility biographies
*****************************************************************************
*****************************************************************************

	cd "$EHDIR"
	use "_index_n.dta", clear
	
**********************************************************
* Section I: Generate flag variables for missings in child order and coresidence
********************************************************

* Aim: order children according to biological children only. For argumentation see Histories-Manual 

* Flag person if at least one birthdate of biological child is not known (no year information)
	*--> (no ordering possible!)
	
		g FLAG_M_bio_dob_=0
		replace FLAG_M_bio_dob_=1 if dobk==-7 & (statusk==1 )
		by id, sort: egen FLAG_M_bio_dob=max(FLAG_M_bio_dob_)
	
		lab var FLAG_M_bio_dob "Respondent with non-recoded missings in birthdate of biological children"
		lab def FLAG_M_bio_dob 0 "No missing" 1 "At least one birthdate of biological child is not known" 
		lab val FLAG_M_bio_dob FLAG_M_bio_dob
		drop FLAG_M_bio_dob_

		* ! respondents with missings in the order of biological children should be dropped for analysis.
	
* Flag if information on coresidence of anchor with kid is not known
	
		* ID 154322000: recoding of coresidence information in 2nd wave to -3 because kid did before.
		replace currlivII=-3 if dodkII>0 & dodkII!=.
			
		g FLAG_M_bioLIVK=0
		g FLAG_M_nonbioLIVK=0
		foreach var of varlist  currlivI currlivII currliv_detailI currliv_detailII livkbegI ///
		livkendI livkbirthI dodkI dodkII {
		replace FLAG_M_bioLIVK=1 if `var' ==-7 & statusk==1 
		replace FLAG_M_nonbioLIVK=1 if `var' ==-7 & statusk!=1 
		}
		lab var FLAG_M_bioLIVK "Information on coresidence with biological child not known"
		lab def FLAG_M_bioLIVK 0 "No missing" 1 "Information on coresidence not known"
		lab val FLAG_M_bioLIVK FLAG_M_bioLIVK
		lab val FLAG_M_nonbioLIVK FLAG_M_bioLIVK

		* respondents with missings in the order of biological children should be dropped for analysis.
			
****************************************************************	
* Section II: Change order according to biological children only 
******************************************************************
	* (non-biol. children are sorted unspecifically after biol. children
		g  bio_dobk = dobk
		replace bio_dobk=9999 if statusk!=1
		replace bio_dobk=8888 if FLAG_M_bio_dob==1
		sort id bio_dobk
		
		g bio_index=.
		by id, sort: replace bio_index=_n 
		replace bio_dobk=-3 if bio_dobk==9999
		replace bio_dobk=-7 if bio_dobk==8888
		lab var bio_dobk "Date of birth of biological child in months since january 1900"
		lab def DNA -3 "-3 Does not apply" -7"-7 Incomplete information"
		lab val bio_dobk DNA
		
		* (-3 "Does not apply") ---- no child/ no biological child
		* (-7 "Incomplete information") ---- if the respondent has at least one biological child without birthdate, 
		* 			all biological children are recoded to (-7).
		
	*Parityspecific information only apply to biological children	
		g bio_sexk=-3
		by id, sort: replace bio_sexk=sexk if bio_dobk>0
		lab var bio_sexk "Sex of biological child"
		lab def sexk2 -7 "-7 Incomplete Information" -3"-3 Does not apply" 1"1 Male" 2"2 Female"
		lab val bio_sexk sexk2
		
		rename index indexk
		lab var indexk "Order of child by age (generated by biochild.do)"
		
		
		g bio_numberk=-3
		by id, sort: replace bio_numberk=number if bio_dobk>0
		lab var bio_numberk "Original child order given in interview (as in anchor$.dta)"
		
		lab def NUMK 	-7 "-7 Incomplete information" ///
						-3 "-3 Does not apply"   ///
						1"1st reported child" ///
						2"2nd reported child" ///
						3"3rd reported child" ///
						4"4th reported child" ///
						5"5th reported child" ///
						6"6th reported child" ///
						7"7th reported child" ///
						8"8th reported child" ///
						9"9th reported child" ///
						10"10th reported child" 
		lab val bio_numberk NUMK
	
	* Overview over child order varibles and their differing reference
		* 1. bio_numberk : from original dataset *Anchor (order of children as stated in interview, here only for biological children)
		*    --> Will be kept in in order to facilitate matching with parity specific data from other datasets.
		* 2. indexk : in biochild.do generated order by considering all childrens date of birth
		*    --> Will be dropped in order to keep clear
		* 3. bio_index : in Eventhistory.do generated order of only biological children (by date of birth), referred to as "parity" of children 
		
	drop indexk sexk number

	cd "$EHDIR"
	save _bioorder_long_wide.dta, replace
			
***************************************************************
* CHAPTER 4: PREPARATION OF VARIABLES FOR THE EPISODE SPLITTING
**********************************************************	

* Note: Birthdates differ between data sets: thus we use one extracted date of birth (from anchor1-3) 
*	for all further time variables


**********************************************************		
*4.1 Partnership biography
************************************************************				
	
	
	use "_ids_dates1_2_3P_DD.dta", clear
	keep id dob
	save "_dob.dta", replace
	
	cd "$EHDIR"
	use "_biopart_flagged.dta", clear
	rename dob dob_BIOPART 
	merge m:1 id using _ids_dates1_2_3P_DD.dta
	drop if _merge!=3
	
	tab dob dob_BIOPART if dob!= dob_BIOPART // 26 ids haben in biopart ein anderes Geburtsdatum als in den Anker1 vermerkt
	drop dob_BIOPART _merge
	
	 

	
*UNION
	gen DUR_beg=beg-dob if beg>0 
	gen DUR_end=end-dob if end>0 
		replace DUR_end=dodp-dob if end==-66
		replace DUR_end=. if end==-99

	local i=0
		while `i'<7  {
		local default1=`i'+1
				capture noisily {
				replace b`default1'beg=-7 if b`default1'end==-7
				replace b`default1'end=-7 if b`default1'beg==-7
				gen U_DUR_`default1'BREAK=b`default1'beg-dob if b`default1'beg>0 
				gen U_DUR_`default1'ENDBREAK=b`default1'end-dob  if b`default1'end>0
				}
		local i=`i'+1
		}
			
*COHAB		
	gen DUR_cohbeg=cohbeg-dob if cohbeg>0 
	gen DUR_cohend=cohend-dob if cohend>0 
		replace DUR_cohend=dodp-dob if cohend==-66
		replace DUR_cohend=. if cohend==-99


	local i=0
		while `i'<7  {
		local default1=`i'+1
				capture noisily {
				replace b`default1'cohbeg=-7 if b`default1'cohend==-7
				replace b`default1'cohend=-7 if b`default1'cohbeg==-7
				gen C_DUR_`default1'BREAK=b`default1'cohbeg-dob if b`default1'cohbeg>0 
				gen C_DUR_`default1'ENDBREAK=b`default1'cohend-dob  if b`default1'cohend>0
				}
		local i=`i'+1
		}			
			
	
	*MARR		
	gen DUR_marbeg=marbeg-dob if marbeg>0 
	gen DUR_marend=marend-dob if marend>0 
		replace DUR_marend=dodp-dob if marend==-66
		replace DUR_marend=. if marend==-99

	
	*Duration death partner
	g DUR_deathP=.
	replace DUR_deathP=dodp-dob if dodp>0
	replace DUR_deathP=9999 if dodp==-7
	
* Drop not needed data
	drop respw2 currw* dodp beg end b1beg b2beg b3beg b4beg b5beg b6beg b7beg ///
	b1end b2end b3end b4end b5end b6end b7end /// 
	cohbeg cohend b1cohbeg b2cohbeg b3cohbeg b1cohend b2cohend b3cohend marbeg marend pid DEMODIFF /*flag_ehc partflag2_hg */


	

*********
*Reshape
		
	reshape wide dobp sexp  marcer /// 
	biopartflag1  biopartflag3 biopartflag4 FLAG_M_UNION ///
	DUR_beg DUR_end  U_DUR_1BREAK U_DUR_1ENDBREAK U_DUR_2BREAK U_DUR_2ENDBREAK /// 
	U_DUR_3BREAK U_DUR_3ENDBREAK U_DUR_4BREAK U_DUR_4ENDBREAK U_DUR_5BREAK U_DUR_5ENDBREAK /// 
	U_DUR_6BREAK U_DUR_6ENDBREAK U_DUR_7BREAK U_DUR_7ENDBREAK ///
	DUR_cohbeg DUR_cohend C_DUR_1BREAK C_DUR_1ENDBREAK C_DUR_2BREAK C_DUR_2ENDBREAK ///
	C_DUR_3BREAK C_DUR_3ENDBREAK FLAG_M_COHAB FLAG_M_MARR DUR_marbeg DUR_marend DUR_deathP ///
	index_m index_c  ///
	 IMP_beg IMP_end IMP_cohbeg IMP_cohend IMP_marbeg IMP_marend pnow1 pnow2 pnow3,  i(id) j(index)

	cd "$EHDIR"
	save "_part1.dta", replace



***************************************************************
*4.2. Fertility biography
****************************************************************

* RECODING and GENERATION of variables as preparation for generation of duration variables
	cd "$EHDIR"
	use _bioorder_long_wide.dta, clear
	tab DEMODIFF, m
	*6 rows =. (BERALL!) 
	*delete these 6 rows
	drop if DEMODIFF==.
	
*Indicator of child born after 1. wave
	g AFTERW1=0
	replace AFTERW1=1 if intdatI==. & bio_dobk>0 & bio_dobk!=.
	
*Indicator of additional nonbiological children after 1. wave
	g AFTERW1_nb=0
	replace AFTERW1_nb=1 if intdatI==. & statusk!=1 & statusk!=.
		
	
* Recode start of living together since birth =1 if start of living together = date of birth or date of birth-1 
* Only refers to children before wave 1 as this information is not available for later children
		
	replace livkbirthI=1 if livkbegI==bio_dobk & livkbegI!=. & livkbegI>0 
	*(169 cases)
	replace livkbirthI=1 if livkbegI==bio_dobk-1 & livkbegI!=. & livkbegI>0
	*(one case)

**********************************************
* Duration variables
**********************************************
	
*GENERAL

	recode intdatw2 .=-3
	recode intdatw2 .a=-3

	recode intdatw3 .=-3
	recode intdatw3 .a=-3

	g DUR_INT1=intdatw1-dob
	g DUR_INT2=intdatw2-dob if intdatw2>0 
	g DUR_INT3=intdatw3-dob if intdatw3>0 
	
* Biological children: being born
	
	*** Duration until birth of biological child in months since birth or anchor***
	 *egen No=rownonmiss(status*)
	generate DUR_BIOKID=bio_dobk-dob if bio_dobk>0 
	lab var  DUR_BIOKID "Months since birth of anchor until birth of bio. child"
	

* Biological children: deceased	
	*Before wave1
	recode dodk* (.=-3)
	generate DUR_DEATHBIOKID=dodkI-dob if bio_dobk>0 & dodkI>0 
	*Before wave2
	replace DUR_DEATHBIOKID=dodkII-dob if bio_dobk>0 & dodkII>0			/* No child died between wave 1 and 2 */
	lab var  DUR_DEATHBIOKID "Months since birth of anchor until death of bio. child"

	*Flag if date of death of child is unknown
	g FLAG_DODK =0
	replace FLAG_DODK=1 if bio_dobk>0 & dodkI==-7 | dodkII==-7
	
* Biological children: moving in/out
	
	*** Duration until start of living with biological child in months since birth or anchor***
		g DUR_MOVBIOKID=.
		
		* Living together since birth
			* Date of start of living together = date of birth, in case of living together since birth.
			*(information only available for children born before wave 1)
			replace DUR_MOVBIOKID=bio_dobk-dob if livkbirthI==1 & bio_dobk>0 
			
			* No information on coresided episodes for deceased children in biochild.dta.
			* Here we assume that the child lived together with the respondent since birth until death
			replace DUR_MOVBIOKID=bio_dobk -dob if dodkI>0 & dodkI!=.  & bio_dobk>0 
			
			* For children born after wave 1 we have no information an coresiding episodes, 
			* here we assume that the respondent was living togetehr with the child since date of birth 
			* in case the respondent lives together with the child at time of 2nd interview
			replace DUR_MOVBIOKID=bio_dobk-dob if (currlivII==1 | currlivII==2 | currlivIII==1 | currlivIII==2) & AFTERW1==1 & bio_dobk>0 
			* If respondent has a child after wave 1 and does not coreside with this child at 2nd resp. 3rd interview
			* he/she is considered to have never lived together with the child (except the child is dead, see here:)
			replace DUR_MOVBIOKID=bio_dobk-dob if currlivII>2 & currlivII!=. & AFTERW1==1 & dodkII>0  & dodkII!=. & bio_dobk>0  // 0 cases 
			replace DUR_MOVBIOKID=bio_dobk-dob if currlivIII>2 & currlivIII!=. & AFTERW1==1 & dodkIII>0  & dodkIII!=. & bio_dobk>0  // 0 cases 

		* Not living together since birth
		* = livkbegI, if this applies. (Information only available for children born before wave 1.)   
		replace DUR_MOVBIOKID=livkbegI-dob if livkbirthI!=1 & AFTERW1==0 & bio_dobk>0 & dodkI<0
		replace DUR_MOVBIOKID=. if DUR_MOVBIOKID<0

	*** Duration until end of living with child in months since birth or anchor***
		g DUR_OUTBIOKID=.
		* Ongoing (in case of earlier interruption it will be replaced below)
		replace DUR_OUTBIOKID=. if currlivII==1 | currlivII==2 & bio_dobk>0
		replace DUR_OUTBIOKID=. if currlivIII==1 | currlivIII==2 & bio_dobk>0

		*= livkendI, if this applies. (Information only available for children born before wave 1.)   
		replace DUR_OUTBIOKID=livkendI-dob if livkendI>0 & livkendI!=. & bio_dobk>0

		* Deceased childen
		replace DUR_OUTBIOKID=dodkI-dob if dodkI>0 & bio_dobk>0
		replace DUR_OUTBIOKID=dodkII-dob if dodkII>0 & bio_dobk>0
		replace DUR_OUTBIOKID=dodkIII-dob if dodkIII>0 & bio_dobk>0

		* Moved out between wave 1 and 2: 
		* Here we assume that the respondent was living togetehr with the child since one month prior to 2nd interview.
		* Random imputation would be more appropriate, but cause unreasonable amount of coding and checks
		replace DUR_OUTBIOKID= intdatII-1-dob if currlivI<=2 & currlivI>0 & currlivII>2 & currlivII!=. & bio_dobk>0
		replace DUR_OUTBIOKID= intdatIII-1-dob if currlivII<=2 & currlivII>0 & currlivIII>2 & currlivIII!=. & bio_dobk>0

	
		* If either the start or the end of the coresidence episode with the child is missing, we recode the whole episode to missing.
		replace DUR_MOVBIOKID=. if FLAG_M_bioLIVK==1
		replace DUR_OUTBIOKID=. if FLAG_M_bioLIVK==1
		
***	
	
* 251 kids which are not biological: consider the period of coresiding with anchor

*Nonbiological children: moving in/out

	*** Duration until start of living with nonbiological child in months since birth or anchor***
		g DUR_MOVnonBIOKID=.
		
		* Living together since birth
			* Date of start of living together = date of birth, in case of living together since birth.
			*(information only available for children born before wave 1)
			replace DUR_MOVnonBIOKID=dobk-dob if livkbirthI==1 & (dobk>0 & dobk!=.) & (statusk!=1 & statusk!=.)
			
			* No information on coresided episodes for deceased children in biochild.dta.
			* Here we assume that the child lived together with the respondent since birth until death		
			replace DUR_MOVnonBIOKID=dobk -dob if dodkI>0 & dodkI!=.  & (dobk>0 & dobk!=.) & (statusk!=1 & statusk!=.)
			
			* For children born after wave 1 we have no information an coresiding episodes, 
			* here we assume that the respondent was living togetehr with the child since date of 2nd interview
			* in case the respondent lives together with the child at time of 2nd interview	
			* Random imputation would be more appropriate, but cause unreasonable amount of coding and checks
			replace DUR_MOVnonBIOKID=intdatII-dob if (currlivII==1 | currlivII==2) & AFTERW1_nb==1  & (statusk!=1 & statusk!=.)
			replace DUR_MOVnonBIOKID=intdatIII-dob if (currlivIII==1 | currlivIII==2) & AFTERW1_nb==1  & (statusk!=1 & statusk!=.)

		* Not living together since birth
		* = livkbegI, if this applies. (Information only available for children born before wave 1.)
		replace DUR_MOVnonBIOKID=livkbegI-dob if livkbirthI!=1 & AFTERW1_nb==0 & (statusk!=1 & statusk!=.) & dodkI<0
		replace DUR_MOVnonBIOKID=. if DUR_MOVnonBIOKID<0
		

	*** Duration until end of living with child in months since birth or anchor***

		g DUR_OUTnonBIOKID=.
		* Ongoing (in case of earlier interruption it will be replaced below)
		replace DUR_OUTnonBIOKID=. if currlivII==1 | currlivII==2  & (statusk!=1 & statusk!=.)
		replace DUR_OUTnonBIOKID=. if currlivIII==1 | currlivIII==2  & (statusk!=1 & statusk!=.)

		*= livkendI fr Vor-Welle1 Kinder, bei denen dies zutrifft
		replace DUR_OUTnonBIOKID=livkendI-dob if livkendI>0 & livkendI!=.  & (statusk!=1 & statusk!=.)

		* Deceased childen
		replace DUR_OUTnonBIOKID=dodkI-dob if dodkI>0 & (statusk!=1 & statusk!=.)
		replace DUR_OUTnonBIOKID=dodkII-dob if dodkII>0 & (statusk!=1 & statusk!=.)
		replace DUR_OUTnonBIOKID=dodkIII-dob if dodkIII>0 & (statusk!=1 & statusk!=.)

		* Moved out between wave 1 and 2: 
		* here we assume that the respondent was living togetehr with the child since one month prior to 2nd interview
		* Random imputation would be more appropriate, but cause unreasonable amount of coding and checks
		replace DUR_OUTnonBIOKID= intdatII-1-dob if currlivI<=2 & currlivI>0 & currlivII>2 & currlivII!=.  & (statusk!=1 & statusk!=.)
		replace DUR_OUTnonBIOKID= intdatIII-1-dob if currlivII<=2 & currlivII>0 & currlivIII>2 & currlivIII!=.  & (statusk!=1 & statusk!=.)

		* If either the start or the end of the coresidence episode with the child is missing, we recode the whole episode to missing.
		replace DUR_MOVnonBIOKID=. if FLAG_M_nonbioLIVK==1
		replace DUR_OUTnonBIOKID=. if FLAG_M_nonbioLIVK==1
	
	******** 		
	**** For a summary of here used assumptions see manual Schnor/Bastin. 
	******** 	
		
	
	drop b1livkendI  b1livkbegI  intdat intdatI* currliv* livkbegI livkendI livkbirthI AFTERW1_nb ///
	dodk* statusk dobk
	
	
	drop if dob==.
	*Only applies to 1 case/line (results from our matching strategy)
	* trifft nun auf keinen mehr zu! (mit 3 Wellen und DemoDiff drin
	
	reshape wide surveykid pnoI pnoII pnoIII ///
	bio_sexk parentid bio_numberk bio_dobk ///
	IMP_dobbiok IMP_dodbiok IMP_beglivbiok IMP_endlivbiok  FLAG_M_bioLIVK IMP_beglivnonbiok IMP_endlivnonbiok  FLAG_M_nonbioLIVK ///
	DUR_BIOKID DUR_DEATHBIOKID FLAG_DODK DUR_MOVBIOKID DUR_OUTBIOKID DUR_MOVnonBIOKID DUR_OUTnonBIOKID AFTERW1, i(id) j(bio_index)


	
		
	
	* Generate Variable that shows bio_index-number of surveyed kid
		g CAPIBIOK=-3
		lab var CAPIBIOK "Order of surveyed bio. child (shows order of younger child in case of more than one surveyed child)"
		local i=0
		while `i'<11 {
		local default =`i'+1

				capture noisily {
				replace CAPIBIOK=`default' if surveykid`default'==1


			}
		local i=`i'+1
		}
	
			
			lab def CAPIBIOK 	-3 "-3 Does not apply" -7 "-7 Incomplete information" ///
								1"1st bio. child was interviewd" 2"2nd bio. child" 			///
								3"3rd bio. child" 4"4th bio. child"				///
								5"5th bio. child" 6"6th bio. child"				///
								7"7th bio. child" 8"8th bio. child"				///
								9"9th bio. child" 10"10th bio. child"	
				
			lab val CAPIBIOK CAPIBIOK
			
	* Recode missings, that occur in reshaping process to -3 "Does not apply"
	* Add variable and value labels that get detached when reshaping
			
		local i=0
		while `i'<11 {
		local default =`i'+1
			capture noisily {
				
				recode bio_sexk`default' .= -3
				lab var bio_sexk`default' "Sex of `default'. bio. child"
				lab val bio_sexk`default' sexk2	
				
				recode bio_numberk`default' .= -3
				lab var bio_numberk`default' "Original order of `default'. bio. as in anchor$ data"
				lab val bio_numberk`default' NUMK
								
				lab var IMP_dobbiok`default' "Imputed month in date of birth of `default'. bio. child"
				lab var IMP_dodbiok`default' "Imputed month in date of death of `default'. bio. child"
				lab var IMP_beglivbiok`default' "Imputed month in start date of coresidence with `default'. bio. child"
				lab var IMP_endlivbiok`default' "Imputed month in end date of coresidence with `default'. bio. child"
			
			}
		local i=`i'+1
		}
	
			
			
	
	foreach var of varlist IMP* {
		lab def `var' 0 "No imputation" 1 "Only year information" 2 "Only season information"
		lab value `var' `var'
		}

			
		drop surveykid*
		
		
	
	cd "$EHDIR"
	save "_fert1.dta", replace


	
	
*******************************************
* CHAPTER 5 GENERATING EVENT HISTORY DATA
*******************************************

******************************************************************************
* Section I: Merging the partnership and the fertility biography 
**************************************************************************************
	
clear all
set mem 1g
set more off


use "$EHDIR\_part1.dta", clear
merge m:1 id using "$EHDIR\_fert1.dta"
drop _merge

				
	* Adapt parent number for each kid to new ordering (partner index) 
		local i=0
				
			while `i'<21 {
			local default1 =`i'+1
				
				g UNIONORDER_BIOK`default1'=-3
				lab def UNIONORDER_BIOK`default1' -7 "Incomplete information" -3 "-3 Does not apply" 97 "97 Another person" 	///
											0 "No partner" 1 "1st Partner"  				///
											2 "2nd Partner" 3 "3rd Partner"  				///
											4 "4th Partner" 5 "5th Partner"  				///
											6 "6th Partner" 7 "7th Partner"  				///
											8 "8th Partner" 9 "9th Partner"  				///
											10 "10th Partner" 11 "11th Partner" 			///
											12 "12th Partner" 13 "13th Partner"  			///
						
				lab val UNIONORDER_BIOK`default1' UNIONORDER_BIOK`default1'
				lab var UNIONORDER_BIOK`default1' "Partner order number of 2nd biological parent of `default1'. bio. child"

				
				local j=0
				while `j'<13 {
				local default2 =`j'+1
					
					capture noisily {
						
					* Before wave 1 kids
					replace  UNIONORDER_BIOK`default1'=`default2' if pnow1`default2'==pnoI`default1' ///
					& pnow1`default2'!=. & bio_dobk`default1'>0
					replace  UNIONORDER_BIOK`default1'=97 if pnoI`default1'==97 & bio_dobk`default1'>0
					replace  UNIONORDER_BIOK`default1'=-7 if pnoI`default1'==-7 & bio_dobk`default1'>0

					* Between wave 1+2 kids
					replace  UNIONORDER_BIOK`default1'=`default2' if pnoI`default1'==. ///
					& pnow2`default2'==pnoII`default1' & pnoII`default1'!=. & bio_dobk`default1'>0
					replace  UNIONORDER_BIOK`default1'=97 if pnoI`default1'==. & pnoII`default1'==97 & bio_dobk`default1'>0
					replace  UNIONORDER_BIOK`default1'=-7 if pnoI`default1'==. & pnoII`default1'==-7 & bio_dobk`default1'>0

					* Between wave 2+3 kids
					replace  UNIONORDER_BIOK`default1'=`default2' if pnoII`default1'==. ///
					& pnow3`default2'==pnoIII`default1' & pnoIII`default1'!=. & bio_dobk`default1'>0				
					replace  UNIONORDER_BIOK`default1'=97 if pnoII`default1'==. & pnoIII`default1'==97 & bio_dobk`default1'>0
					replace  UNIONORDER_BIOK`default1'=-7 if pnoII`default1'==. & pnoIII`default1'==-7 & bio_dobk`default1'>0

					}

				local j=`j'+1
				}

			local i=`i'+1
				}
			
			
			
			drop bio_dobk*
			
			cd "$EHDIR"
			save "_allhistories_temp1.dta", replace



*********************************************************
* Section II: Stset the data
******************************************************

	
	clear all
	set mem 2g
	set maxvar 30000
	set more off

	cd "$EHDIR"
	use "_allhistories_temp1.dta", clear

* Generating TIME VARIABLE

	g TEST=0
	recode intdatw2 .=-3
	recode intdatw3 .=-3

	* _t0 = birthdate of anchor person
		g TIME=intdatw1+1-dob 
		replace TIME=intdatw2+1-dob if intdatw2>0
		replace TIME=intdatw3+1-dob if intdatw3>0 
	* Add one month to have a split for month of interview 
	* -> Otherwise events in the month of interview are not taken into account
	* -> makes separation of prospective and retrospective information easier

	stset TIME, fail(TEST) id (id)


	
	
******************************************************
* 5.1. General time varying variables (AGEANC at the end - saves running time)
******************************************************
						

	*VAR (INT): Information comes from which wave?
	stsplit INT1, at (0 1) after (time=DUR_INT1)
	g INT=INT1
	recode INT -1=0 0=1 1=2
	
	stsplit INT2, at (0 1) after (time=DUR_INT2)


	stsplit INT3, at (0 1) after (time=DUR_INT3)


	replace INT=3 if INT2==0
	
	replace INT=4 if INT3==-1 & INT2>0
	replace INT=5 if INT3==0

	*correct if there was a third, but no second wave
	replace INT=0 if INT1==-1 & DUR_INT3>0 & DUR_INT2==.
	replace INT=1 if INT1== 0 & DUR_INT3>0 & DUR_INT2==.
	replace INT=99 if INT1==1 & DUR_INT3>0 & DUR_INT2==.
	replace INT=5 if INT1==1 & INT3==0 & DUR_INT3>0 & DUR_INT2==.


	lab var INT "Before/at/after resp. interview"
	lab def INT 0 "Before 1. Interview" 1"Month of 1st Interview" 2"Between wave 1 and 2" 3"Month of 2nd Interview" ///
				4 "Between wave 2 and 3" 5"Month of 3rd interview" 99"Between wave 1 and 3"
	lab val INT INT
	
	* Note: for DemoDiff the second Interview is considered as second interview, although the survey design refers to the second interview as "wave 2/3".
	
	drop DUR_INT* INT2 INT3
	
	
************************************
* 5.2 Time varying partnership information
************************************

	*(divided into 2 sections:
         *Section I: episode splitting
		 *Section II: summarize time-variant variables)
		 
***********************************************************
* Section I: Episode splitting
*********************************************************

	* VAR (UNION START): TVUNION_{unionorder}

		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVUNION_`default1', at (0) after (time=DUR_beg`default1')
				replace TVUNION_`default1'=3 if DUR_beg`default1'==.
				recode TVUNION_`default1' -1=0 0=1 3=0
				lab var TVUNION_`default1' "In `default1'. union or not"
				label define TVUNION_`default1'  0 "No partner" 1 "In union" 
				label value TVUNION_`default1' TVUNION_`default1'
				}
			local i=`i'+1
			}
		
  * VAR (UNION END): TVSEP_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVSEP_`default1', at (0) after (time=DUR_end`default1')
				replace TVSEP_`default1'=3 if DUR_end`default1'==.
				recode TVSEP_`default1' -1=0 0=1 3=0
				lab var TVSEP_`default1' "`default1'. union dissolved or not"
				label define TVSEP_`default1'  1 "Separated" 
				label value TVSEP_`default1' TVSEP_`default1'
				}
			local i=`i'+1
			}
			
		drop DUR_beg* DUR_end*
			
	* VAR(START OF UNION BREAKS): TVU_{breakorder}BREAK_{unionorder}
		
		*break
		local i=0
			while `i'<13  {
			local default1=`i'+1
			local j=0
			   while `j'<7 {
			   local default2=`j'+1
				   capture noisily {
				   stsplit TVU_`default2'BREAK_`default1', at (0) after (time=U_DUR_`default2'BREAK`default1')
				   replace TVU_`default2'BREAK_`default1'=3 if U_DUR_`default2'BREAK`default1'==.
				   recode TVU_`default2'BREAK_`default1' -1=0 0=1 3=0
				   lab var TVU_`default2'BREAK_`default1' "In `default2'. break within `default1'. union or not"
				   label define TVU_`default2'BREAK_`default1'  1 "Union break" 
				   label value TVU_`default2'BREAK_`default1' TVU_`default2'BREAK_`default1'
				   }
			   local j=`j'+1
			   }
			local i=`i'+1
			}
		
		*VAR (END OF UNION BREAK): TVU_{breakorder}ENDBREAK_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
			   local j=0
			   while `j'<7 {
			   local default2=`j'+1
				   capture noisily {
				   stsplit TVU_`default2'ENDBREAK_`default1', at (0) after (time=U_DUR_`default2'ENDBREAK`default1')
				   replace TVU_`default2'ENDBREAK_`default1'=3 if U_DUR_`default2'ENDBREAK`default1'==.
				   recode TVU_`default2'ENDBREAK_`default1' -1=0 0=1 3=0
				   lab var TVU_`default2'ENDBREAK_`default1' "`default2'. break within `default1'. union finished or not"
				   label define TVU_`default2'ENDBREAK_`default1'  1 "End union break" 
				   label value TVU_`default2'ENDBREAK_`default1' TVU_`default2'ENDBREAK_`default1'
				   }
			   local j=`j'+1
			   }
			local i=`i'+1
			}
			
		drop U_DUR*

	*VAR: (START OF COHABITATION): TVCOHAB_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVCOHAB_`default1', at (0) after (time=DUR_cohbeg`default1')
				replace TVCOHAB_`default1'=3 if DUR_cohbeg`default1'==.
				recode TVCOHAB_`default1' -1=0 0=1 3=0
				lab var TVCOHAB_`default1' "In `default1'. cohabitation or not"
				label define TVCOHAB_`default1' 0 "Not cohabiting" 1 "Cohabiting" 
				label value TVCOHAB_`default1' TVCOHAB_`default1'
				}
			local i=`i'+1
			}
	
	drop DUR_cohbeg*

	*VAR: (START OF COHABITATION BREAK): TVC_{breakorder}BREAK_{unionorder}
		
		*break
		local i=0
			while `i'<13  {
				local default1=`i'+1
				local j=0
				while `j'<4 {
				local default2=`j'+1
					capture noisily {
					stsplit TVC_`default2'BREAK_`default1', at (0) after (time=C_DUR_`default2'BREAK`default1')
					replace TVC_`default2'BREAK_`default1'=3 if C_DUR_`default2'BREAK`default1'==.
					recode TVC_`default2'BREAK_`default1' -1=0 0=1 3=0
					lab var TVC_`default2'BREAK_`default1' "In `default2'. break within `default1'. cohabitation or not"
					label define TVC_`default2'BREAK_`default1'  1 "Cohab break" 
					label value TVC_`default2'BREAK_`default1' TVC_`default2'BREAK_`default1'
					}
				local j=`j'+1
				}
			local i=`i'+1
			}
	
		
		*VAR (END OF COHABITATION BREAK): TVC_{breakorder}ENDBREAK_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
			   local j=0
			   while `j'<4 {
			   local default2=`j'+1
				   capture noisily {
				   stsplit TVC_`default2'ENDBREAK_`default1', at (0) after (time=C_DUR_`default2'ENDBREAK`default1')
				   replace TVC_`default2'ENDBREAK_`default1'=3 if C_DUR_`default2'ENDBREAK`default1'==.
				   recode TVC_`default2'ENDBREAK_`default1' -1=0 0=1 3=0
				   lab var TVC_`default2'ENDBREAK_`default1' "`default2'. break within `default1'. cohabitation finished or not"
				   label define TVC_`default2'ENDBREAK_`default1'  1 "End cohab break" 
				   label value TVC_`default2'ENDBREAK_`default1' TVC_`default2'ENDBREAK_`default1'
				   }
			   local j=`j'+1
			   }
			local i=`i'+1
			}


	*VAR (END OF COHABITATION): TVCOHABEND_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVCOHABEND_`default1', at (0) after (time=DUR_cohend`default1')
				replace TVCOHABEND_`default1'=3 if DUR_cohend`default1'==.
				recode TVCOHABEND_`default1' -1=0 0=1 3=0
				lab var TVCOHABEND_`default1' "`default1'. cohabitation dissolved or not"
				label define TVCOHABEND_`default1'  1 "Not cohabiting" 
				label value TVCOHABEND_`default1' TVCOHABEND_`default1'
				}
			local i=`i'+1
			}
			
		drop C_DUR_* DUR_cohend*			
										  
	*VAR (START OF MARRIAGE): TVMARR_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVMARR_`default1', at (0) after (time=DUR_marbeg`default1')
				replace TVMARR_`default1'=3 if DUR_marbeg`default1'==.
				recode TVMARR_`default1' -1=0 0=1 3=0 
				lab var TVMARR_`default1' "In `default1'. marriage or not"
				label define TVMARR_`default1' 0 "Single" 1 "Married" 2 "Divorced"
				label value TVMARR_`default1' TVMARR_`default1'
				}
			local i=`i'+1
			}
		 
		 		  
	*VAR (END OF MARRIAGE=DIVORCE): TVDIV_{unionorder}
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVDIV_`default1', at (0) after (time=DUR_marend`default1')
				replace TVDIV_`default1'=3 if DUR_marend`default1'==.
				recode TVDIV_`default1' -1=0 0=1 3=0
				lab var TVDIV_`default1' "`default1'. marriage dissolved or not"
				label define TVDIV_`default1'  1 "Divorced" 
				label value TVDIV_`default1' TVDIV_`default1'
				}
			local i=`i'+1
			}
			
		drop DUR_marbeg* DUR_marend*	
			
		sort id _t0 

**************************************************
* Section II: Summarize time-variant variables
***************************************************

*UNION
* UNION-STATUS becomes time-varying; for each union 

g helpFLAG=0
local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace helpFLAG=`default1' if  TVUNION_`default1'==1 
			}
			local i=`i'+1
			}


	*before: TVUNION_{1,13}: single(0)-union(1) - union(1)
	*now: TVUNION_{1,13}: single(0)-union(1)-single(0)
		local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace TVUNION_`default1'=0 if TVSEP_`default1'==1
				}
			local i=`i'+1
			}


	* one variable for ALL unions (UNION)
	* before: TVUNION_{1,13}: single(0)-union(1)-single(0)
	* now: UNION: single(0)-union(1)-single(0)-union(1)-single(0)
		g UNION=0
		local i=0
			 while `i'<13 {
			 local default1=`i'+1
				 capture noisily {
				 replace UNION=1 if  TVUNION_`default1'==1
				 }
			 local i=`i'+1
			 }
			 
	lab var UNION "Union status"
	lab define UNION 0 "No partner" 1 "In union"
	lab value UNION UNION


	*generate Unionorder 
		g UNIONORDER=0
	 
		local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace UNIONORDER=`default1' if TVUNION_`default1'==1 & TVSEP_`default1'==0
				}
			local i=`i'+1
			}
		lab var UNIONORDER "Order of unions (shows order of later partners in simultaneous cases"	
		
		lab val UNIONORDER UNIONORDER_BIOK1
		
		*for highest order (13), because there is no sep date)
		*replace UNIONORDER=13 if TVUNION_13==1

*************************************************
*including union breaks 

*BREAK

* BREAK becomes time-varying; for each union break 

	* before: {1,7}union break {1,13}: no break(0) - break(1) - break(1)
	* now: {1,7}union break {1,13}: no break(0) - break(1) - no break(0)
	* ! default2=break order / default1=union order 
		local i=0
			while `i'<13 {
			local default1=`i'+1
				local j=0
				while `j'<7 {
				local default2 = `j'+1
					capture noisily {
					replace TVU_`default2'BREAK_`default1'=0 if TVU_`default2'ENDBREAK_`default1'==1
					}
				local j=`j'+1
				}
			local i=`i'+1
			}

	* one variable for each union (recode the different breaks within one union)
	* before: {1,7}union break {1,13}: no break(0) - break(1) - no break(0)
	* now: union break{1,13}: no break (0) - break(1) - no break(0)- break(1) - no break(0)
		local i=0 
			 while `i'<13 {
			 local default1=`i'+1
				capture noisily {
				 g U_BREAK`default1'=TVU_1BREAK_`default1'
				 replace U_BREAK`default1'=1 if TVU_1BREAK_`default1'==0 & TVU_2BREAK_`default1'==1
				 replace U_BREAK`default1'=1 if TVU_2BREAK_`default1'==0 & TVU_3BREAK_`default1'==1
				 replace U_BREAK`default1'=1 if TVU_3BREAK_`default1'==0 & TVU_4BREAK_`default1'==1
				 replace U_BREAK`default1'=1 if TVU_4BREAK_`default1'==0 & TVU_5BREAK_`default1'==1
				 replace U_BREAK`default1'=1 if TVU_5BREAK_`default1'==0 & TVU_6BREAK_`default1'==1
				 replace U_BREAK`default1'=1 if TVU_6BREAK_`default1'==0 & TVU_7BREAK_`default1'==1
				 *lab var U_BREAK`default1' "Break in `default1'. union"
				 lab define U_BREAK`default1' 1 "Union break"
				 label value U_BREAK`default1' U_BREAK`default1'
				 }
			 local i=`i'+1
			 }
			 
	 *label variable --> Does not work within loop above
		local i=0 
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				lab var U_BREAK`default1' "Break in `default1'. union"
				}
			local i=`i'+1
			}
			
	*one variable for ALL unions and ALL breaks
	* before: union break{1,6}
	* now: union break: no break(0) - break(1) - no break(0) - break(1) - no break(0)  
		
		
		g U_BREAK=U_BREAK1
		forvalues x=2/13 {
		local xm1= `x'-1
		cap replace U_BREAK=1 if U_BREAK`xm1'==0 & U_BREAK`x'==1
						}


		lab var U_BREAK "Union break status"

	* For those with simultaneous unions: generate second union order which shows the order of the second partner "UNIONORDER_SIM"
	* UNIONORDER: shows the later partner in simultaneous cases
	* UNIONORDER_SIM: shows the "previous" partner in simultaneous cases
	* Ex: 0 1 1 2 2 2 3 3 3 (UNIONORDER)
	*     0 1 1 2 2 2 2 3 3 (UNIONORDER_SIM)
	*                 x      (simultaneous partnerships)
	* UNIONORDER_SIM can be used as a flag variable for simultaneous unions (UNIONORDER_SIM!=0)


sort id

	*BREAKORDER: counts breaks within one union
		g UBREAKORDER=0
		lab var UBREAKORDER "Order of union breaks within one union"
		lab def UBREAKORDER 0 "No break" 1 "1st union break" 2 "2nd union break" 3 "3rd union break" 4 "4th union break" ///
							5 "5th union break" 6 "6th union break" 7 "7th union break" 8 "8th union break"
		lab val UBREAKORDER UBREAKORDER

		local i=0
			while `i'<13 {
			local default1=`i'+1
			local j=0
				while `j'<7 {
				local default2 = `j'+1
					capture noisily {
					replace UBREAKORDER=`default2' if TVU_`default2'BREAK_`default1'==1 & TVU_`default2'ENDBREAK_`default1'==0
					}
				local j=`j'+1
				}
			local i=`i'+1
			}



		g UNIONORDER_SIM=0
		local i=0
			while `i'<13 {
			local default1=`i'+1
				local j=0
				while `j'<13 {
					local default2 = `j'+1
					capture noisily {
					replace UNIONORDER_SIM=`default1' if TVUNION_`default1'==1 & TVUNION_`default2'==1 & `i'<`j' 
					}
				local j=`j'+1
				}
			local i=`i'+1
			}

		lab var UNIONORDER_SIM "Union order for simultaneous unions: shows order of 'previous' partner"
		lab def UNIONORDER_SIM 	0 "No partner" 1 "1st partner" ///
								2 "2nd partner" 3 "3rd partner" ///
								4 "4th partner" 5 "5th partner" ///
								6 "6th partner" 7 "7th partner" ///
								8 "8th partner" 9 "9th partner" ///
								10 "10th partner" 11 "11th partner" ///
								12 "12th partner" 13 "13th partner"
		lab val UNIONORDER_SIM UNIONORDER_SIM


	* UNION should be zero (no partner) when  a union break and no other partnership occurs
		replace UNION=0 if U_BREAK==1 & UNIONORDER_SIM==0

	* UNIONORDER should be zero when  a union break occurs and no other partnership occurs
		replace UNIONORDER=0 if U_BREAK==1 & UNION==0
	
	* UNIONORDER_SIM should be zero when a union break occurs and other partnerships occur
		replace UNIONORDER_SIM=0 if UBREAKORDER==1

		
		*HOMOSEXUAL PARTNERSHIP EPISODES
		
		g HOMOSEX=0
		lab def HOMOSEX 0 "No union" 1 "Heterosexual union" 2 "Homosexual union"
		lab val HOMOSEX HOMOSEX
		forvalues x=1/13 {
		replace HOMOSEX=1 if UNION==1 & (UNIONORDER==`x'| UNIONORDER_SIM==`x') & (sex!=sexp`x')
		replace HOMOSEX=2 if UNION==1 & (UNIONORDER==`x'| UNIONORDER_SIM==`x') & (sex==sexp`x')
		lab var HOMOSEX "Sexual orientation within union"
		}

	*drop variables
	drop TVU* TVS* U_*

	
	

*************************************************************************
*COHABITATION
*************************************************************************

*COHAB-STATUS becomes time-varying; for each cohabitation

	* before: TVCOHAB_{1;13}: not cohabiting (0)- cohabiting(1)- cohabiting(1)   
	* now: TVCOHAB_{1;13}: not cohabiting(0) - cohabiting(1) - not cohabiting(0)

	
	local i=0
		while `i'<13 {
		local default1=`i'+1
			capture noisily {
			replace TVCOHAB_`default1'=0 if TVCOHABEND_`default1'==1
			}
		local i=`i'+1
		}

	* one variable for ALL cohabitations (COHAB)
	* before: TVCOHAB_{1;13}: not cohabiting(0) - cohabiting(1) - not cohabiting(0)
	* now: COHAB: not cohabiting(0) - cohabiting(1) - not cohabiting(0)- cohabiting(1) - not cohabiting(0)
		g COHAB=0
		label define COHAB 0 "Not in cohabitation " 1 "Cohabiting"
		label value COHAB COHAB
		lab var COHAB "Cohabitation status"

		local i=0
			 while `i'<13 {
			 local default1=`i'+1
				 capture noisily {
				 replace COHAB=1 if TVCOHAB_`default1'==1 
				 }
			local i=`i'+1
			}

			
	*generate COHABORDER
		g COHABORDER=0
		label var COHABORDER "Order of cohabitations (shows additional/'later' cohabitation partners in simultaneous cases)" 
	 
		local i=0
			while `i'<13 {
			local default1=`i'+1 
				capture noisily {
				replace COHABORDER=index_c`default1' if TVCOHAB_`default1'==1 
				}
			local i=`i'+1
			}
	
	lab val COHABORDER index_c
	
	
	* Union order of cohabiting partner = states which unionpartner the respondent is cohabiting with
		g UNIONORDER_COHAB=0
		lab var UNIONORDER_COHAB "Union order number of cohabiting partner"
		lab val UNIONORDER_COHAB UNIONORDER
		lab def UNIONORDER_COHAB 	0"No domestic partner" ///
									1"1st partner is domestic partner" ///
									2"2nd partner is domestic partner" ///
									3"3rd partner is domestic partner" ///
									4"4th partner is domestic partner" ///
									5"5th partner is domestic partner" ///
									6"6th partner is domestic partner" ///
									7"7th partner is domestic partner" ///
									8"8th partner is domestic partner" ///
									9"9th partner is domestic partner" ///
									10"10th partner is domestic partner" ///
									11"11th partner is domestic partner" ///
									12"12th partner is domestic partner" ///
									13"13th partner is domestic partner" ///
							
		lab val UNIONORDER_COHAB UNIONORDER_COHAB							
		
		local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace UNIONORDER_COHAB=`default1' if TVCOHAB_`default1'==1 
				}
			local i=`i'+1
			}
		
	
***************************************
* Including cohabitation breaks


*BREAK

* BREAK becomes time-varying; for each cohabitation break 

	* before: {1,3}cohab break {1,13}: no break(0) - break(1) - break(1)
	* now: {1,3}cohab break {1,13}: no break(0) - break(1) - no break(0)
	* ! default2=break order / default1=union order 
		local i=0
			while `i'<13 {
			local default1=`i'+1
				local j=0
				while `j'<4 {
				local default2 = `j'+1
					capture noisily {
					replace TVC_`default2'BREAK_`default1'=0 if TVC_`default2'ENDBREAK_`default1'==1
					}
				local j=`j'+1
				}
			local i=`i'+1
			}



	* one variable for each cohabitation (account for the different breaks within one cohabitation)
	* before: {1,3}cohab break {1,13}: no break(0) - break(1) - no break(0)
	* now: cohab break{1,13}: no break (0) - break(1) - no break(0)- break(1) - no break(0)
		local i=0 
			 while `i'<13 {
			 local default1=`i'+1
				
		 g C_BREAK`default1'=TVC_1BREAK_`default1'
		cap lab var C_BREAK`default1' "Break in `default1'. cohabitation"
		cap lab define C_BREAK`default1' 1 "Cohab break"
		cap lab value C_BREAK`default1' C_BREAK`default1'
		
		local j=0 
			 while `j'<4 {
			 local default2=`j'+1
				capture noisily {
			
				 replace C_BREAK`default1'=1 if TVC_`default1'BREAK_`default1'==0 & TVC_`default2'BREAK_`default1'==1
				}
				local j=`j'+1
				}
			local i=`i'+1
			}
		 
		 	 
	 
	* one variable for ALL cohabitations and ALL breaks
	* before: cohab break{1,6}
	* now: union break: no break (0) - break(1) - no break(0)- break(1) - no break(0)
		g C_BREAK=C_BREAK1
		forvalues x=2/13 {
		local xm1= `x'-1
		cap replace C_BREAK=1 if C_BREAK`xm1'==0 & C_BREAK`x'==1
		}
		 lab var C_BREAK "Cohabitation break status"

	* !!?? nach dem Rekodieren scheint es nur noch cohabitation breaks in erster und zweiter Kohabitation zu geben!!??


	* For those with simultaneous cohabitations: generate second cohab order which shows the order of the second cohabiting partner "COHABORDER_SIM"
	* COHABORDER: shows the later COHAB partner in simultaneous cases
	* COHABORDER_SIM: shows the "previous" COHAB partner in simultaneous cases
	* Ex: 0 1 1 2 2 2 3 3 3 (COHABORDER)
	*     0 1 1 2 2 2 2 3 3 (COHABORDER_SIM)
	*                 x      (simultaneous cohabitations)
	* COHABORDER_SIM can be used as a flag variable for simultaneous unions (UNIONORDER_SIM!=0)

	sort id
	

	
	g COHABORDER_SIM=0
	local i=0
		while `i'<13 {
		local default1=`i'+1
		local j=0
			while `j'<13 {
			local default2 = `j'+1
				capture noisily {
				replace COHABORDER_SIM=index_c`default1' if TVCOHAB_`default1'==1 & TVCOHAB_`default2'==1 & `i'<`j'
				}
			local j=`j'+1
			}
		local i=`i'+1
		}

	lab var COHABORDER_SIM "Cohabitation order for simultaneous cohabitations: shows order of 'previous' cohabiting partner"
	
	lab val COHABORDER_SIM index_c

	* COHAB should be zero (no partner in the household) when a cohabitation break occurs and no other partner enters the household
		replace COHAB=0 if C_BREAK==1 & COHABORDER_SIM==0

	* COHABORDER should be zero (no partner) when a cohabitation break occurs and no other partner enters the household
		replace COHABORDER=0 if C_BREAK==1 & COHAB==0

	*Order of cohabitation (C_BREAKORDER): counts breaks within one cohabitation
		g CBREAKORDER=0
		lab var CBREAKORDER "Order of cohabitation breaks within one cohabitation"
		lab def CBREAKORDER 0 "No break" 1 "1st cohab break" 				///
							2 "2nd cohab break" 3 "3rd cohab break" 		///
							4 "4th cohab break" 5 "5th cohab break" 		///
							6 "6th cohab break" 7 "7th cohab break" 		///
							8 "8th cohab break"
		lab val CBREAKORDER CBREAKORDER

		local i=0
			while `i'<13 {
			local default1=`i'+1
			local j=0
				while `j'<7 {
				local default2 = `j'+1
					capture noisily {
					replace CBREAKORDER=`default2' if TVC_`default2'BREAK_`default1'==1 & TVC_`default2'ENDBREAK_`default1'==0
					}
				local j=`j'+1
				}
			local i=`i'+1
			}
		
* COHABORDER_SIM should be zero when  a union break occurs and other partnerships occur
		replace COHABORDER_SIM=0 if CBREAKORDER==1
		
		
		
	drop TVC* C_*
		
		
*********************
*MARRIAGE		
*********************


	*MARRIAGE-STATUS becomes time-varying: not married - married - divorced
	*eine Variable fuer alle Marriages
	*Indicator for Marriage order

	local i=0
		while `i'<13 {
		local default1=`i'+1
			capture noisily {
			replace TVMARR_`default1'=2 if TVDIV_`default1'==1
			}
		local i=`i'+1
		}

	generate MARR=TVMARR_1
	
	lab var MARR "Marriage status"
	label define MARR 0 "Single" 1 "Married" 2 "Divorced"
	label value MARR MARR
	
	local i=1
		while `i'<13 {
		local default1=`i'+1
		
			capture noisily {
			replace MARR=1 if TVMARR_`default1'==1
			replace MARR=2 if TVMARR_`default1'==2
			}
		local i=`i'+1
		}
	
	
	* marriage order = ordering of all marriages
		g MARRORDER=0
		lab var MARRORDER "Order of marriages"
		local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace MARRORDER=index_m`default1' if TVMARR_`default1'==1 & (TVDIV_`default1'==0 )
				}
			local i=`i'+1
			}
			
		lab val MARRORDER index_m
	
	* union order of married partner = states which partner the respondent is married to
		g UNIONORDER_MARR=0
		lab var UNIONORDER_MARR "Union order number of married partner"
		lab val UNIONORDER_MARR UNIONORDER
		lab def UNIONORDER_MARR 	0"Not married " ///
									1"1st partner is spouse" ///
									2"2nd partner is spouse" ///
									3"3rd partner is spouse" ///
									4"4th partner is spouse" ///
									5"5th partner is spouse" ///
									6"6th partner is spouse" ///
									7"7th partner is spouse" ///
									8"8th partner is spouse" ///
									9"9th partner is spouse" ///
									10"10th partner is spouse" ///
									11"11th partner is spouse" ///
									12"12th partner is spouse" ///
									13"13th partner is spouse" ///
							
		lab val UNIONORDER_MARR UNIONORDER_MARR		
		
		
		local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace UNIONORDER_MARR=`default1' if TVMARR_`default1'==1 & (TVDIV_`default1'==0 ) 
				}
			local i=`i'+1
			}

	
		drop TVM* TVDIV_*

	
	* marcer* becomes timevariant
		sort id _t0
		
		g MARCER=-3
		
		local i=0
			while `i'<13 {
			local default=`i'+1
				capture noisily {
				replace MARCER=marcer`default' if MARR==1 & UNIONORDER==`default'
				}
			local i=`i'+1
			}
	
	label val MARCER marcer
	lab var MARCER "Type of wedding ceremony"
	
	drop marcer*

	
	*States whether a partner dies while the partnership lasts
		
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit FLAG_deathP_`default1' if DUR_deathP`default1'!=9999, at (0) after (time=DUR_deathP`default1')
				replace FLAG_deathP_`default1'=3 if DUR_deathP`default1'==. 
				replace FLAG_deathP_`default1'=2 if DUR_deathP`default1'==9999
				recode FLAG_deathP_`default1' -1=0 0=1 3=0 
				lab var FLAG_deathP_`default1' "`default1'. partner died"
				label define FLAG_deathP_`default1'  0 "No death of partner" 1 "Partner died" 2 "Incomplete information"
				label value FLAG_deathP_`default1' FLAG_deathP_`default1'
				}
			local i=`i'+1
			}


	g DEADPARTNER=0
	lab def DEADPARTNER 	-7 "-7 Incomplete information" 99 "More than one partner died" ///
							0 "No death of partner" 1 "1st partner died" 	///
							2 "2nd partner died" 3 "3rd partner died"		///
							4 "4th partner died" 5 "5th partner died"		///   
							6 "6th partner died" 7 "7th partner died" 		///
							8 "8th partner died" 9 "9th aprtner died" 		///	
							10 "10th partner died" 11 "11th partner died"	///
							12 "12th partner died" 13 "13th partner died"	
	
	lab var DEADPARTNER "Death of partner"
	lab val DEADPARTNER DEADPARTNER 
	 

		*local i=1
			local j=0
			
				*while `i'<21 {
				*local default2=`i'+1
					while `j'<11 {
					local default1=`j'+1
						capture noisily {
					
						replace DEADPARTNER=`default1' if FLAG_deathP_`default1'==1 
						* if several partners died, we code the flag variable as 99
						*--> Does not work correctly --> commented out
						*replace DEADPARTNER=99 if FLAG_deathP_`default1'==1 & FLAG_deathP_`default2'==1 
						* if information is unknown
						replace DEADPARTNER=-7 if FLAG_deathP_`default1'==2
						
						}
					local j=`j'+1
					*}
				*local i=`i'+1
				}
				
	
	
	

	cd "$EHDIR"
	save _unionhistories.dta, replace
	

****************************************************************
* Section III: Time-varying flag variables
**********************************************************

********************************************************************
* III.a Flag variables that indicate imputed dates 
**************************************************************************

	g IMP_UNION=0
	forvalues x=1/13 {
	cap replace IMP_UNION=1 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_beg`x'==1
	cap replace IMP_UNION=2 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_beg`x'==2
	}
	lab var IMP_UNION "Imputed union start date"
	lab def IMP_UNION 0 "No imputation" 1 "Only year information" 2 "Only season information"
	lab val IMP_UNION IMP_UNION
	
	g IMP_SEP=0
	forvalues x=1/13 {
	cap replace IMP_SEP=1 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_end`x'==1
	cap replace IMP_SEP=2 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_end`x'==2
	}
	lab var IMP_SEP "Imputed union end date"
	lab def IMP_SEP 0 "No imputation" 1 "Only year information" 2 "Only season information"
	lab val IMP_SEP IMP_SEP

g IMP_COHAB=0
	forvalues x=1/13 {
	cap replace IMP_COHAB=1 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_cohbeg`x'==1
	cap replace IMP_COHAB=2 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_cohbeg`x'==2
	}
	
	replace IMP_COHAB=0 if COHABORDER==0
	
	sort id COHABORDER
	rename IMP_COHAB IMPCOHAB
	by id COHABORDER: egen IMP_COHAB=max(IMPCOHAB)
	lab var IMP_COHAB "Imputed cohabitation start date"
	lab def IMP_COHAB 0 "No imputation" 1 "Only year information" 2 "Only season information"
	lab val IMP_COHAB IMP_COHAB
	drop IMPCOHAB
	
g IMP_COHABend=0
	forvalues x=1/13 {
	cap replace IMP_COHABend=1 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_cohend`x'==1
	cap replace IMP_COHABend=2 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_cohend`x'==2
	}
	
	replace IMP_COHABend=0 if COHABORDER==0
	
	sort id COHABORDER
	rename IMP_COHABend IMPCOHABend
	by id COHABORDER: egen IMP_COHABend=max(IMPCOHABend)
	lab var IMP_COHABend "Imputed cohabitation end date"
	lab def IMP_COHABend 0 "No imputation" 1 "Only year information" 2 "Only season information"
	lab val IMP_COHABend IMP_COHABend
	drop IMPCOHABend

g IMP_MARR=0
	forvalues x=1/13 {
	cap replace IMP_MARR=1 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_marbeg`x'==1
	cap replace IMP_MARR=2 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_marbeg`x'==2
	}
	
	replace IMP_MARR=0 if MARRORDER==0
	
	sort id MARRORDER
	rename IMP_MARR IMPMARR
	by id MARRORDER: egen IMP_MARR=max(IMPMARR)
	lab var IMP_MARR "Imputed wedding date"
	lab def IMP_MARR 0 "No imputation" 1 "Only year information" 2 "Only season information"
	lab val IMP_MARR IMP_MARR
	drop IMPMARR
	
g IMP_DIV=0
	forvalues x=1/13 {
	cap replace IMP_DIV=1 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_marend`x'==1
	cap replace IMP_DIV=2 if (UNIONORDER==`x'| UNIONORDER_SIM==`x') & IMP_marend`x'==2
	}
	
	replace IMP_DIV=0 if MARRORDER==0
	
	sort id MARRORDER
	rename IMP_DIV IMPDIV
	by id MARRORDER: egen IMP_DIV=max(IMPDIV)
	lab var IMP_DIV "Imputed divorce date"
	lab def IMP_DIV 0 "No imputation" 1 "Only year information" 2 "Only season information"
	lab val IMP_DIV IMP_DIV
	drop IMPDIV
		
		order IMP_beg*, sequential
		order IMP_end*, sequential
		order IMP_cohbeg*, sequential
		order IMP_cohend*, sequential
		order IMP_marbeg*, sequential
		order IMP_marend*, sequential
		
	drop 	IMP_beg1-IMP_beg13 IMP_end1-IMP_end13 IMP_cohbeg1-IMP_cohbeg13 				///
			IMP_cohend1-IMP_cohend13 IMP_marbeg1-IMP_marbeg13 IMP_marend1-IMP_marend13 	
			


***************************************************************
*III.b biopartflag variables
********************************************************
	
	foreach var of varlist biopartflag* FLAG* IMP*{
			replace `var'=0 if `var'==. 
		}
	
	
g biopartflag1=0
		local i=0
			while `i'<13 {
			local default=`i'+1
				capture noisily {
				replace biopartflag1=1 if biopartflag1`default'==1 & UNIONORDER==`default'
				}
			local i=`i'+1
			}
			
	
g biopartflag3=0
		local i=0
			while `i'<13 {
			local default=`i'+1
				capture noisily {
				replace biopartflag3=1 if biopartflag3`default'==1 & UNIONORDER==`default'
				}
			local i=`i'+1
			}

g biopartflag4=0
		local i=0
			while `i'<13 {
			local default=`i'+1
				capture noisily {
				replace biopartflag4=1 if biopartflag4`default'==1 & UNIONORDER==`default'
				}
			local i=`i'+1
			}
	
	g BIOPARTFLAG=0
		replace BIOPARTFLAG=1 if biopartflag1==1
		replace BIOPARTFLAG=3 if biopartflag3==1
		replace BIOPARTFLAG=4 if biopartflag4==1
		lab var BIOPARTFLAG "Flag inconsistencies in partnership biography (biopart)"
		lab def BIOPARTFLAG 	0 "No inconsistency" 									///
								1 "Marriage earlier than beginning of relationship" 	///
								3 "Beginning current and end previous marriage" 		///
								4 "Year of birth partner", modify						
		lab val BIOPARTFLAG BIOPARTFLAG

	drop biopartflag1* biopartflag3* biopartflag4*

***************************************************************
*III.c Check variables that indicate overlapping union and cohabitation episodes
***************************************************************************
	
* check UNIONORDER_SIM variable with imputed cases and recode episodes which overlap due to imputations
*********************

rename _t0 START
rename _t END

	sort id START
	by id: g A=START if UNIONORDER_SIM!=0 & (UNIONORDER_SIM[_n-1]==0 | UNIONORDER_SIM[_n-1]!=UNIONORDER_SIM)
	by id:g B=END if UNIONORDER_SIM!=0 & (UNIONORDER_SIM[_n+1]==0 | UNIONORDER_SIM[_n+1]!=UNIONORDER_SIM)
	sort id  UNIONORDER_SIM UNIONORDER 
	by id  UNIONORDER_SIM UNIONORDER: egen AA=max(A)
	by id  UNIONORDER_SIM UNIONORDER: egen BB=max(B)

	g TESTa=.
	replace TESTa=BB-AA if UNIONORDER_SIM!=0 & IMP_UNION!=0 | IMP_SEP!=0

	g NO_SIM=0
	replace NO_SIM=1 if UNIONORDER_SIM!=0 &  IMP_SEP==1 & TESTa<13
	replace NO_SIM=2 if UNIONORDER_SIM!=0 &  IMP_UNION==1 & TESTa<13
	replace NO_SIM=3 if UNIONORDER_SIM!=0 &  IMP_SEP==2 & TESTa<4
	replace NO_SIM=4 if UNIONORDER_SIM!=0 &  IMP_UNION==2 & TESTa<4

	sort id START
* one id has two times simultaneous unions with the same partners: manual imputation
	replace AA=A if id==87191000 & START==286
	replace BB=B if id==87191000 & START==286

	by id: replace START=AA[_n-1] if UNIONORDER_SIM==0 & UNIONORDER_SIM[_n-1]!=0 & NO_SIM[_n-1]!=0 & AA[_n-1]!=.

	drop if NO_SIM!=0

	drop TESTa A AA B BB NO_SIM 


*check COHABORDER_SIM variable with imputed cases and recode episodes which overlap due to imputations
*********************
	sort id START
	by id: g A=START if COHABORDER_SIM!=0 & COHABORDER_SIM[_n-1]==0
	by id:g B=END if COHABORDER_SIM!=0 & COHABORDER_SIM[_n+1]==0
	sort id  COHABORDER_SIM
	by id  COHABORDER_SIM: egen AA=max(A)
	by id  COHABORDER_SIM: egen BB=max(B)

	g TESTa=.
	replace TESTa=BB-AA if COHABORDER_SIM!=0 & IMP_COHAB!=0 | IMP_COHABend!=0

	g NO_SIM=0
	replace NO_SIM=1 if COHABORDER_SIM!=0 &  IMP_COHABend==1 & TESTa<13
	replace NO_SIM=2 if COHABORDER_SIM!=0 &  IMP_COHAB==1 & TESTa<13
	replace NO_SIM=3 if COHABORDER_SIM!=0 &  IMP_COHABend==2 & TESTa<4
	replace NO_SIM=4 if COHABORDER_SIM!=0 &  IMP_COHAB==2 & TESTa<4

	sort id START
	by id: replace START=AA[_n-1] if COHABORDER_SIM==0 & COHABORDER_SIM[_n-1]!=0 & NO_SIM[_n-1]!=0 & AA[_n-1]!=.

	drop if NO_SIM!=0

	drop TESTa A AA B BB NO_SIM 


******************************************
* III.d Flag variables (missing information that has not been imputed)
*******************************************


* union episodes
*******************
	sort id
	
	g FLAG_M_UNION=0
	lab var FLAG_M_UNION "Missing union episode"
	lab define FLAG_M_UNION 0 "No missing" 1 "Missing union episode" 2 "Missing union break episode", replace
	lab val FLAG_M_UNION FLAG_M_UNION

	* first union episode is missing:
	by id: replace FLAG_M_UNION=1 if UNIONORDER==0 & helpFLAG==0 & FLAG_M_UNION1==1

	* if several union episodes are missing: 
	forvalues x=2/13 {
				capture noisily {
				by id: replace FLAG_M_UNION=1 if UNIONORDER==0 & helpFLAG==0 & FLAG_M_UNION1==1 & FLAG_M_UNION`x'==1
				}
			}
			
	* if a higher order union episode is missing:
	forvalues x=2/13 {
			local xm1= `x'-1
			capture noisily {
			by id: replace  FLAG_M_UNION=1 if UNIONORDER==0 & helpFLAG==`xm1' & FLAG_M_UNION`x'==1
						}
				}
						
	* if a union break episode is missing:					
	forvalues x=1/13 {
				capture noisily {
				by id: replace FLAG_M_UNION=2 if UNIONORDER==`x' &  FLAG_M_UNION`x'==1 
				}
			}
	
	forvalues x=1/13 {
			local xm1= `x'-1
			capture noisily {
			by id: replace  FLAG_M_UNION=0 if UNIONORDER==0 & helpFLAG==`xm1' & FLAG_M_UNION`x'[_n+1]==2
						}
					}
					
			sort id helpFLAG
			by id helpFLAG: egen help_miss=min(FLAG_M_UNION)
			
			replace FLAG_M_UNION=0 if help_miss==0 & FLAG_M_UNION!=2
	
	order FLAG_M_UNION*, sequential
	drop  help_miss FLAG_M_UNION1-FLAG_M_UNION13
	
						
* cohabitation episodes
************************************************************************************************
						
sort id					
g FLAG_M_COHAB=0
lab var FLAG_M_COHAB "Missing cohabitation episode"
	lab define FLAG_M_COHAB 0 "No missing" 1 "Missing cohabitation episode", replace
	lab val FLAG_M_COHAB FLAG_M_COHAB

	* if the cohabitation episode is missing:
		forvalues x=1/13 {
				capture noisily {
				by id: replace FLAG_M_COHAB=1 if FLAG_M_COHAB`x'==1 & UNIONORDER==`x' & UNION==1
				by id: replace FLAG_M_COHAB=1 if COHAB==0 & FLAG_M_COHAB`x'==1 & FLAG_M_UNION==1
				}
			}
			
	order FLAG_M_COHAB*, sequential
	drop FLAG_M_COHAB1-FLAG_M_COHAB13


* marriage episodes
************************************************************************
sort id 
g FLAG_M_MARR=0
lab var FLAG_M_MARR "Missing marriage episode"
	lab define FLAG_M_MARR 0 "No missing" 1 "Missing marriage episode", replace
	lab val FLAG_M_MARR FLAG_M_MARR
	
	* If the cohabitation episode is missing:
		forvalues x=1/13 {
				capture noisily {
				by id: replace FLAG_M_MARR=1 if FLAG_M_MARR`x'==1 & UNIONORDER==`x' & UNION==1
				by id: replace FLAG_M_MARR=1 if MARR==0 & FLAG_M_MARR`x'==1 & (FLAG_M_UNION==1 | FLAG_M_COHAB==1)
				}
			}

	order FLAG_M_MARR*, sequential
	drop FLAG_M_MARR1-FLAG_M_MARR13
	

	save "$EHDIR\_PARTSPLIT.dta",replace
	
	
	
****************************************************************
* 5.3. Time-varying fertility information
****************************************************************
	set more off
	use "$EHDIR\_PARTSPLIT.dta",clear

	
	rename START _t0
	rename END _t
	
*Age of child 
	local i=0
			while `i'<21 {
			local default =`i'+1
			capture noisily {
				stsplit AGEBIOK`default', after at(-9 0 12 24 36 48 60 72 84 96 108 120 		///
				132 144 156 168 170 182 194 206 218 230 242 254) (time=DUR_BIOKID`default')
				
				replace AGEBIOK`default'=999 if DUR_BIOKID`default'==.
				replace AGEBIOK`default'=-7 if FLAG_M_bio_dob==1
				recode AGEBIOK`default'  -10=0 -9=1 0=2 12=3 24=4 36=5 48=6 60=7 72=8 84=9 	 	///
				96=10 108=11 120=12 132=13 144=14 156=15 168=16 170=17 182=18 194=19 206=20 	///
				218=21 230=22 242=23 254=24 999=0
				
				lab var AGEBIOK`default' "Age of `default'. bio. kid"
				label define AGEBIOK`default' 	 0 "Childless"									///				
												 1 "Pregnant" 									///
												 2 "0 years old" 								///
												 3 "1 year old" 								///
												 4 "2 years old" 								///
												 5 "3 years old" 								///
												 6 "4 years old" 								///
												 7 "5 years old" 								///
												 8 "6 years old" 								///
												 9 "7 years old" 								///
												 10 "8 years old" 								///
												 11 "9 years old" 								///
												 12 "10 years old"								///
												 13 "11 years old"								///
												 14 "12 years old"								///
												 15 "13 years old"								///
												 16 "14 years old"								///
												 17 "15 years old"								///
												 18 "16 years old"								///
												 19 "17 years old"								///
												 20 "18 years old"								///
												 21 "19 years old"								///
												 22 "20 years old"								///
												 23 "21 years old"								///
												 24 "22 years and older"						///
												 -7 "-7 Incompl. information"							
				label value AGEBIOK`default' AGEBIOK`default'
				}
				local i=`i'+1
				}


	*Age of youngest child
	  	g AGEBIOK_YNG=AGEBIOK1
		local i=1
			while `i'<21 {
			local default1 = `i'+1
						capture noisily {

				replace AGEBIOK_YNG=AGEBIOK`default1' if AGEBIOK`default1'<AGEBIOK_YNG &  AGEBIOK`default1'!=0
				}
				local i=`i'+1
			    }
		lab var AGEBIOK_YNG "Age of youngest bio. kid"
		label value  AGEBIOK_YNG AGEBIOK1

	save _CHECK, replace 
		use _CHECK, clear 

	* Biological children died or not
		*Each child
		local i=0
				while `i'<13  {
				local default1=`i'+1
					
					capture noisily {
					by id, sort: egen FLAG_DODK_`default1'=max(FLAG_DODK`default1')
					stsplit DEADBIOK_`default1', at (0) after (time=DUR_DEATHBIOKID`default1')
					replace DEADBIOK_`default1'=3 if DUR_DEATHBIOKID`default1'==.
					replace DEADBIOK_`default1'=-7 if DUR_DEATHBIOKID`default1'==. & FLAG_DODK_`default1'==1
					recode DEADBIOK_`default1' -1=0 0=1 3=0
					lab var DEADBIOK_`default1' "`default1'. bio. child died"
					label define DEADBIOK_`default1' 	-7 "-7 Incomplete information" ///
														 0 "Child did not die" 1 "Child died" 
					label value DEADBIOK_`default1' DEADBIOK_`default1'
					}
				local i=`i'+1
				}
		
		*Summarized in one variable for all children
		g DEADBIOK=0
		lab var DEADBIOK "Decease of which child"
		lab def DEADK 	0"No child died"	-7 "-7 Incomplete information"			///
						1"1st child died" 2"2nd child died" 3"3rd child died"	///
						4"4th child died" 5"5th child died" 6"6th child died"	///
						7"7th child died" 8"8th child died" 9"9th child died"	///
						10 "10th child died" 99"More than one child died"									
						
		lab val DEADBIOK DEADK	
			
		g FLAG_DODK	= 0
		
						
		*local i=1
			local j=0
			
				*while `i'<21 {
				*local default2=`i'+1
				while `j'<11 {
				local default1=`j'+1
					capture noisily {
					
					replace DEADBIOK=`default1' if DEADBIOK_`default1'==1
					* if several children died, we recode the flag variable to 99 --> Does not work correctly! Commented out
					* replace DEADBIOK=99 if DEADBIOK_`default1'==1  & DEADBIOK_`default2'==1 
					replace FLAG_DODK=1 if FLAG_DODK`default1'==1
						}
				local j=`j'+1
				*	}
				*local i=`i'+1
				}
							
			replace DEADBIOK=-7 if FLAG_DODK==1
	
	
	
	* Living with biological child or not - Time varying

			local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit LIVBIOK`default1', at (0) after (time=DUR_MOVBIOKID`default1')
				replace LIVBIOK`default1'=3 if DUR_MOVBIOKID`default1'==.
				replace LIVBIOK`default1'=-7 if DUR_MOVBIOKID`default1'==. & FLAG_M_bioLIVK`default1'==1
				recode LIVBIOK`default1' -1=0 0=1 3=0
				lab var LIVBIOK`default1' "Coresidence with `default1'. bio. child"
				label define LIVBIOK`default1' 	-7 "-7 Incomplete information" ///
												 0 "Not living with child" 1 "Living with child" 
				label value LIVBIOK`default1' LIVBIOK`default1'
				}
			local i=`i'+1
			}

			
			
		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TV_bioLIVKEND_`default1', at (0) after (time=DUR_OUTBIOKID`default1')
				replace TV_bioLIVKEND_`default1'=3 if DUR_OUTBIOKID`default1'==.
				replace TV_bioLIVKEND_`default1'=-7 if DUR_OUTBIOKID`default1'==. & FLAG_M_bioLIVK`default1'==1
				recode TV_bioLIVKEND_`default1' -1=0 0=1 3=0
				lab var TV_bioLIVKEND_`default1' "Ended living with `default1'. bio. child or not"
				label define TV_bioLIVKEND_`default1' -7 "-7 Incomplete information" 0 "Not ended living with child" 1 "Ended living with child" 
				label value TV_bioLIVKEND_`default1' TV_bioLIVKEND_`default1'
				}
			local i=`i'+1
			}
	
			
		local i=0
			while `i'<13 {
			local default=`i'+1
				capture noisily {
				replace LIVBIOK`default'=0 if TV_bioLIVKEND_`default'==1
				replace LIVBIOK`default'=-7 if TV_bioLIVKEND_`default'==-7
				
				}
			local i=`i'+1
			}
		

	* The following general variable of living together with any biological children 
	* will later on be replaced by the variable indicating whether the respondent is living together with any biological or non-biological children
	* one variable for ALL biological children 
		generate bioLIVK=0
		local i=0
			 while `i'<13 {
			 local default1=`i'+1
				 capture noisily {
				 replace bioLIVK=1 if  LIVBIOK`default1'==1
				 replace bioLIVK=-7 if LIVBIOK`default1'==-7
				 }
			 local i=`i'+1
			 }
			 
	lab var bioLIVK "Coresidence with biological children"
	lab define bioLIVK -7 "-7 Incomplete information" 0 "not living with bio. child" 1 "living with bio. child"
	lab value bioLIVK bioLIVK
	

	* flag variable that marks imputations in the coresidence with children
	local i=0
			while `i'<21 {
			local default=`i'+1
				capture noisily {
				replace IMP_beglivbiok`default'=0 if TV_bioLIVK`default'==0
				replace IMP_endlivbiok`default'=0 if TV_bioLIVK`default'==0
				 }
			 local i=`i'+1
			 }
			 
				


			 
	*NONBIOLOGICAL CHILDREN: EPISODES OF CORESIDING WITH ANCHOR
	*******************************************************
	
	* Living with nonbiological child or not - Time varying
	*! the nonbiological children are not ordered correctly, as the child order refers to biological children only

		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVnonbioLIVK_`default1', at (0) after (time=DUR_MOVnonBIOKID`default1')
				replace TVnonbioLIVK_`default1'=3 if DUR_MOVnonBIOKID`default1'==.
				replace TVnonbioLIVK_`default1'=-7 if DUR_MOVnonBIOKID`default1'==. & FLAG_M_nonbioLIVK`default1'==1
				recode TVnonbioLIVK_`default1' -1=0 0=1 3=0
				lab var TVnonbioLIVK_`default1' "Coresidence with `default1' nonbio. child or not"
				label define TVnonbioLIVK_`default1'  	-7 "-7 Incomplete information" 		///
														0 "Not living with nonbio. child" 	///
														1 "Living with nonbio child" 
				label value TVnonbioLIVK_`default1' TVnonbioLIVK_`default1'
				}
			local i=`i'+1
			}

		local i=0
			while `i'<13  {
			local default1=`i'+1
				capture noisily {
				stsplit TVnonbioLIVKEND_`default1', at (0) after (time=DUR_OUTnonBIOKID`default1')
				replace TVnonbioLIVKEND_`default1'=3 if DUR_OUTnonBIOKID`default1'==.
				replace TVnonbioLIVKEND_`default1'=-7 if DUR_OUTnonBIOKID`default1'==. & FLAG_M_nonbioLIVK`default1'==1
				recode TVnonbioLIVKEND_`default1' -1=0 0=1 3=0
				lab var TVnonbioLIVKEND_`default1' "Ended living with `default1' nonbio. child or not"
				label define TVnonbioLIVKEND_`default1'  -7 "-7 Incomplete information"  0 "Not ended living with nonbio child" 1 "Ended living with nonbio child" 
				label value TVnonbioLIVKEND_`default1' TVnonbioLIVKEND_`default1'
				}
			local i=`i'+1
			}
			

		* Bringing together information of start and end of living together with each child 
		local i=0
			while `i'<13 {
			local default1=`i'+1
				capture noisily {
				replace TVnonbioLIVK_`default1'=0 if TVnonbioLIVKEND_`default1'==1
				replace TVnonbioLIVK_`default1'=-7 if TVnonbioLIVKEND_`default1'==-7
				}
			local i=`i'+1
			}
		
	
	* one variable for ALL non-biological children
		generate LIVKnonbio=0
		local i=0
			 while `i'<13 {
			 local default1=`i'+1
				 capture noisily {
				 replace LIVKnonbio=1 if  TVnonbioLIVK_`default1'==1
				 replace LIVKnonbio=-7 if  TVnonbioLIVK_`default1'==-7
				 }
			 local i=`i'+1
			 }
			 
	lab var LIVKnonbio "Coresidence with nonbiological children"
	lab define LIVKnonbio 	-7 "-7 Incomplete information" 		///
							0 "Not living with nonbio. child" 	///
							1 "Living with nonbio. child"
	lab value LIVKnonbio LIVKnonbio
	
	
	*COMBINATION VARIABLE: living with either biological or/and nonbiological children
	
	g LIVKIDS=.
	replace LIVKIDS=0 if bioLIVK==0 & LIVKnonbio==0
	replace LIVKIDS=1 if bioLIVK==1 & LIVKnonbio==0
	replace LIVKIDS=2 if bioLIVK==0 & LIVKnonbio==1
	replace LIVKIDS=3 if bioLIVK==1 & LIVKnonbio==1
	replace LIVKIDS=-7 if bioLIVK==-7 | LIVKnonbio==-7
	lab var LIVKIDS "Coresidence with biological and/or nonbiological children"
	lab define LIVKIDS 	-7 "-7 Incomplete information" 							///
							0 "Living without children" 							///
							1 "Living only with biological children" 				///
							2 "Living only with nonbiological children" 			///
							3 "Living with biological and nonbiological children"	
	lab val LIVKIDS LIVKIDS
	
	

	 * flag variable that marks imputations in the coresidence with children
	generate IMP_BEGLIVnonbiok=0
	generate IMP_ENDLIVnonbiok=0
	lab val IMP_BEGLIVnonbiok IMP_beglivnonbiok1
	lab var IMP_BEGLIVnonbiok "Imputed month in the start of coresidence with non-biological children"
	lab val IMP_ENDLIVnonbiok IMP_endlivnonbiok1
	lab var IMP_ENDLIVnonbiok "Imputed month in the end of coresidence with non-biological children"
	
	local i=0
			while `i'<21 {
			local default=`i'+1
				capture noisily {
				replace IMP_BEGLIVnonbiok=1 if IMP_beglivnonbiok`default'==1 & TVnonbioLIVK_`default'==1
				replace IMP_ENDLIVnonbiok=1 if IMP_endlivnonbiok`default'==1 & TVnonbioLIVK_`default'==1
				 }
			 local i=`i'+1
			 }

*drop variables
drop LIVKnonbio TVn* TV_bioLIVKEND_* bioLIVK DUR_BIOKID* DUR_MOV* DUR_OUT*  DUR_DEATHBIOKID* DUR_deathP* AFTERW1* FLAG_M_bio_dob* DEADBIOK_*
drop  FLAG_M_bioLIVK* FLAG_M_nonbioLIVK*  IMP_endlivnonbiok* IMP_beglivnonbiok* FLAG_DODK* FLAG_deathP_*
	
		

		rename _t0 START
		lab var START "Beginning of episode in months since birth of anchor"
		rename _t END
		lab var END "End of episode in months since birth of anchor"


	* Make some more variables time variant
	
	* PID* becomes timevariant
	* No pid if Partner is retrospective or date is unknown (-7)
		sort id START
		
		g UORDER_INT1_=0
		replace UORDER_INT1_=UNIONORDER if INT==1 
		by id, sort: egen UORDER_INT1=max(UORDER_INT1)
		
		g UORDER_INT2_=0
		replace UORDER_INT2_=UNIONORDER if INT==3 
		by id, sort: egen UORDER_INT2=max(UORDER_INT2)
		
		g pid=.
		replace pid=PIDI if UNIONORDER==UORDER_INT1
		replace pid=PIDII if UNIONORDER==UORDER_INT2
		lab var pid "Person number partner; if he/she was a partner at interview"
			
			
		* Sex, original number of child and number of surveyed kid: only shown after child was conceived

		
		local i=0
			 while `i'<21 {
			 local default1=`i'+1
				 capture noisily {
				
				* Sex, original number of child and number of surveyed kid: only shown after child was conceived
					
					g SEXBIOK`default1'=bio_sexk`default1'
					replace SEXBIOK`default1'=-3 if AGEBIOK`default1'==0 
					replace SEXBIOK`default1'=-7 if AGEBIOK`default1'==-7
					lab val SEXBIOK`default1' sexk2
					lab var SEXBIOK`default1' "Sex of `default1'. bio. child"
					
					g NUMBERBIOK`default1'=bio_numberk`default1'
					replace NUMBERBIOK`default1'=-3 if AGEBIOK`default1'==0 
					replace NUMBERBIOK`default1'=-7 if AGEBIOK`default1'==-7
					lab val NUMBERBIOK`default1' NUMK
					lab var NUMBERBIOK`default1' "Original order number of `default1'. bio. child in raw data"
					
					replace UNIONORDER_BIOK`default1'=-3 if AGEBIOK`default1'==0
					replace UNIONORDER_BIOK`default1'=-7 if AGEBIOK`default1'==-7
					lab val UNIONORDER_BIOK`default1' UNIONORDER_BIOK`default1'
					lab var UNIONORDER_BIOK`default1' "Partner number of 2nd biol. parent of `default1'. bio. child"
					
					replace CAPIBIOK=-3 if CAPIBIOK==`default1' & AGEBIOK`default1'==0
					replace CAPIBIOK=-7 if AGEBIOK`default1'==-7
					replace CAPIBIOK=-3 if DEMODIFF==1
					lab val CAPIBIOK CAPIBIOK
				
				}
			 local i=`i'+1
			 }
	
	*Adapt pairfam-id-variable-names to facilitate merging with multi-actor data sets
		rename CID cid
		rename MID mid
		rename FID fid
		rename SMID smid
		rename SFID sfid
	
	recode pid cid mid fid smid sfid (.a=.)	
	
	drop UORDER* PIDI PIDII dobp* pno* parentid* bio_sexk* bio_numberk* 

	save "$EHDIR\_PARTFERTSPLIT.dta", replace
	
	
*[end of CHAPTER 5]
***********************************************************************************



	use "$EHDIR\_PARTFERTSPLIT.dta", clear

	
	rename START _t0
	rename END _t


	drop TEST
	g TEST=0
	
	*VAR (AGEANC): Age of anchor person (one split after each birthday)
	stsplit AGEANC, after at (168 (12)max) (time=TEST)
	recode AGEANC	 0=0 168=14 180=15 192=16 204=17 216=18 228=19 240=20 ///
					252=21 264=22 276=23 288=24 300=25 312=26 324=27 ///
					336=28 348=29 360=30 372=31 384=32 396=33 408=34 ///
					420=35 432=36 444=37 456=38 468=39 480=40
	lab var AGEANC "Age of anchor"
	lab def AGEANC 	0 "Below 14 years old" 	///
					14 "14 years old"		///
					15 "15 years old"		///
					16 "16 years old"		///
					17 "17 years old"		///
					18 "18 years old"		///
					19 "19 years old"		///
					20 "20 years old"		///
					21 "21 years old"		///
					22 "22 years old"		///
					23 "23 years old"		///
					24 "24 years old"		///
					25 "25 years old"		///
					26 "26 years old"		///
					27 "27 years old"		///
					28 "28 years old"		///
					29 "29 years old"		///
					30 "30 years old"		///
					31 "31 years old"		///
					32 "32 years old"		///
					33 "33 years old"		///
					34 "34 years old"		///
					35 "35 years old"		///
					36 "36 years old"		///
					37 "37 years old"		///
					38 "38 years old"		///
					39 "39 years old"		///
					40 "40 years old"		///
					41 "41 years old"		///
					42 "42 years old"		///
					43 "43 years old"		///
					44 "44 years old"		///
					45 "45 years old"		///
					46 "46 years old"		
		lab val AGEANC AGEANC
	



lab var DEMODIFF "Respondent of DemoDiff survey"
lab def DEMODIFF 0"No DemoDiff Resp." 1"DemoDiff Resp."
lab val DEMODIFF DEMODIFF




keep 	FLAG_M_MARR FLAG_M_COHAB FLAG_M_UNION IMP_endlivbiok1 IMP_endlivbiok2 IMP_endlivbiok3 IMP_endlivbiok4 ///
		IMP_endlivbiok5 IMP_endlivbiok6 IMP_endlivbiok7 IMP_endlivbiok8 IMP_endlivbiok9 IMP_endlivbiok10 ///
		IMP_beglivbiok1 IMP_beglivbiok2 IMP_beglivbiok3 IMP_beglivbiok4 IMP_beglivbiok5 IMP_beglivbiok6 ///
		IMP_beglivbiok7 IMP_beglivbiok8 IMP_beglivbiok9 IMP_beglivbiok10 id  sex cid mid fid smid sfid dob ///
		IMP_dobbiok1 IMP_dodbiok1 IMP_dobbiok2 IMP_dodbiok2 IMP_dobbiok3 IMP_dodbiok3 IMP_dobbiok4 IMP_dodbiok4 ///
		IMP_dobbiok5 IMP_dodbiok5 IMP_dobbiok6 IMP_dodbiok6 IMP_dobbiok7 IMP_dodbiok7 IMP_dobbiok8 IMP_dodbiok8 ///
		IMP_dobbiok9 IMP_dodbiok9 IMP_dobbiok10 IMP_dodbiok10 DEMODIFF CAPIBIOK UNIONORDER_BIOK1 ///
		UNIONORDER_BIOK2 UNIONORDER_BIOK3 UNIONORDER_BIOK4 UNIONORDER_BIOK5 UNIONORDER_BIOK6 UNIONORDER_BIOK7 ///
		UNIONORDER_BIOK8 UNIONORDER_BIOK9 UNIONORDER_BIOK10 UNIONORDER_BIOK11 UNIONORDER_BIOK12 _t _t0 ///
		INT UNION UNIONORDER UBREAKORDER UNIONORDER_SIM HOMOSEX COHAB COHABORDER UNIONORDER_COHAB COHABORDER_SIM ///
		CBREAKORDER MARR MARRORDER UNIONORDER_MARR MARCER DEADPARTNER IMP_UNION IMP_SEP IMP_COHAB IMP_COHABend ///
		IMP_MARR IMP_DIV BIOPARTFLAG AGEBIOK1 AGEBIOK2 AGEBIOK3 AGEBIOK4 AGEBIOK5 AGEBIOK6 AGEBIOK7 AGEBIOK8 AGEBIOK9 ///
		AGEBIOK10 AGEBIOK_YNG DEADBIOK LIVBIOK1 LIVBIOK2 LIVBIOK3 LIVBIOK4 LIVBIOK5 LIVBIOK6 LIVBIOK7 LIVBIOK8 LIVBIOK9 ///
		LIVBIOK10 LIVKIDS IMP_BEGLIVnonbiok IMP_ENDLIVnonbiok pid SEXBIOK1 NUMBERBIOK1 SEXBIOK2 NUMBERBIOK2 SEXBIOK3 ///
		NUMBERBIOK3 SEXBIOK4 NUMBERBIOK4 SEXBIOK5 NUMBERBIOK5 SEXBIOK6 NUMBERBIOK6 SEXBIOK7 NUMBERBIOK7 SEXBIOK8 ///
		NUMBERBIOK8 SEXBIOK9 NUMBERBIOK9 SEXBIOK10 NUMBERBIOK10 AGEANC
		
		
**************************************
* ORDER OF VARIABLES  
**************************************

* last
 
*Id-Variables of anchor's parents and CAPI kid
 order sfid
 order smid
 order fid
 order mid
  
*Flag variables

 order IMP_ENDLIVnonbiok
 order IMP_BEGLIVnonbiok
 order IMP_endlivbiok*, sequential                                             
 order IMP_beglivbiok*, sequential 
 order IMP_dodbiok*, sequential                                             
 order IMP_dobbiok*, sequential                 
 order IMP_DIV
 order IMP_MARR
 order IMP_COHABend  
 order IMP_COHAB  
 order IMP_SEP
 order IMP_UNION
 
  
 order FLAG_M_MARR 
 order FLAG_M_COHAB 
 order FLAG_M_UNION 
 
 order BIOPARTFLAG

*Fertility History
 order cid
 order DEADBIOK
 order NUMBERBIOK*, sequential
 order SEXBIOK*, sequential
 
 order CAPIBIOK
 order UNIONORDER_BIOK*, sequential
 
 order LIVKIDS
 order LIVBIOK*, sequential
 order AGEBIOK*, sequential

*Partnership History
 order MARCER
 order UNIONORDER_MARR
 order MARR*, sequential

 order UNIONORDER_COHAB
 order CBREAKORDER
 order COHABO*
 order COHAB

 order DEADPARTNER
 order HOMOSEX
 order pid					/* needs to be checked again*/
 order UBREAKORDER
 order UNIONORDER_SIM
 order UNIONORDER
 order UNION
 
*General variables
 order INT
 order AGEANC

 rename _t0 START
 rename _t END
 
 order id START END sex dob DEMODIFF, first


*drop index_c* index_m* intdatw* TEST TIME sexp* helpFLAG b4cohbeg b4cohend INT1 respw3  /*_st _d */


foreach var of varlist  IMP* FLAG*{
			count if `var' ==0
			if r(N) == _N drop `var'
		}




compress
save "$EHDIR\Eventhistory.dta", replace
	
	
	

* delete interim data sets
***********************************
	* cd "$EHDIR"
	/* 
	capture noi erase IMP.dta
	capture noi erase biopart1_v2_reshaped.dta
	capture noi erase COHAB_LONG.dta
	capture noi erase MARR_LONG.dta
	capture noi erase biopart2_index.dta
	capture noi erase ids_dates1.dta
	capture noi erase ids_dates2.dta
	capture noi erase ids_dates1_2.dta
	capture noi erase biochild_long_wide.dta
	capture noi erase IMP_biochild.dta
	capture noi erase index_n.dta
	capture noi erase biopart_flagged.dta
	capture noi erase biopart_long_wide.dta
	capture noi erase part1.dta
	capture noi erase fert1.dta
	capture noi erase histories_temp1.dta
	