#' @title Feedback Class
#' 
#'
#' @description
#' This is the base class for feedback environment objects like [BTM], [Thurstone] and [MSLR].
#'


Feedback = R6Class("Feedback",
                  public = list(
                    
                    #' @field weights (`numeric()`)\cr
                    #' The weight vector determining the utilities of the arms (given the context).
                    weights = NULL,
                    
                    #' @field time_horizon (`integer()`)\cr
                    #' Stores the regret suffered by the learner over the time.
                    time_horizon = NULL,
                    
                    #' @field context_matrices (`list()`)\cr
                    #' Stores the context matrices of the problem instance.
                    context_matrices = NULL,
                    
                    #' @field pert_dis (`function()`)\cr
                    #' the perturbation distribution.
                    pert_dis = NULL,
                    
                    #' @field pert_utilities (`list()`)\cr
                    #' Stores the perturbated utilities leading to the feedback.
                    pert_utilities = list(),
                    
                    #' @field utilities (`list()`)\cr
                    #' Stores the ground-truth utilities.
                    utilities = list(),
                    
                    #' @field opt_arms_utilities (`list()`)\cr
                    #' Stores the choices made by the learner over the time.
                    opt_arms_utilities = list(),
                    
                    #' @field num_arms (`integer()`)\cr
                    #' The number of arms.
                    num_arms = NULL,
                    
                    #' @field dim (`integer()`)\cr
                    #' The dimensionality of the weight vector.
                    dim = NULL,
                    
                    #' @description
                    #' Creates a new instance of this [R6][R6::R6Class] class.
                    #'
                    #' Note that this object is typically constructed via a derived classes, e.g., [BTM], [Thurstone] and [MSLR].
                    initialize = function(weights, time_horizon,  pert_dis, context_matrices =NULL) {
                      
                      self$weights            = weights
                      self$time_horizon       = time_horizon
                      self$context_matrices   = context_matrices
                      self$pert_dis           = pert_dis
                      
                      self$dim                = length(weights)
                      
                      if (!is.null(context_matrices)){
                        # context matrices are provided, i.e., contextual setting.
                        self$num_arms               = ncol(context_matrices[[1]]) 
                        for (i in 1:time_horizon){
                          self$utilities[[i]]       = t(context_matrices[[i]])%*%self$weights
                          self$pert_utilities[[i]]  = self$utilities[[i]] +  pert_dis(self$num_arms)
                        }
                        
                      }
                      else{
                        # context matrices are not provided, i.e., non-contextual setting.
                        self$num_arms           = length(weights) 
                        for (i in 1:time_horizon){
                          self$utilities[[i]]       = self$weights
                          self$pert_utilities[[i]]  = self$weights +  pert_dis(self$num_arms)
                        }
                      }
                      
                    },
                    
                    
                    #' @description
                    #' Printer.
                    #' @param ... (ignored).
                    print = function() { 
                      cat("  Weights: ", self$weights, "\n", sep = "")
                      cat("  time horizon: ", self$time_horizon, "\n", sep = "")
                      cat("  number of arms: ", self$num_arms, "\n", sep = "")
                    },
                    
                    #' @description
                    #' 
                    #' Returns the time horizon of the problem instance. 
                    #'
                    #' @return integer
					#'
                    getTimeHorizon = function() {
                      return (self$time_horizon)
                    },
                    
                    #' @description
                    #' Returns the number of arms of the problem instance.
                    #'
                    #' 
                    #'
                    #' @return integer
                    #'  
                    getNumberArms = function() {
                      return (self$num_arms)
                    },
                    
                    #' @description
                    #' Returns the dimensionality of the problem instance.
                    #'
                    #'
                    #' @return integer
                    #' 
                    getDim = function() {
                      return (self$dim)
                    },
                    
                    #' @description
                    #' Returns the weight vector of the problem instance.
                    #'
                    #' 
                    #'
                    #' @return integer
					#'
                    getWeight = function() {
                      return (self$weights)
                      },
                    
                    #' @description
                    #' Returns the context matrix of the problem instance at some specific timestep.
                    #'
                    #' @param timestep (`integer(1)`)\cr.
                    #' 
                    #'
                    #' @return matrix.
                    getContext = function(timestep) {
                      return (self$context_matrices[[timestep]])
                    },
                    
                    #' @description
                    #' Returns the ground-truth utility of a specific arm at a specific timestep.
                    #'
                    #' @param arm (`integer(1)`)\cr.
                    #' @param timestep (`integer(1)`)\cr.
                    #' 
                    #'
                    #' @return numeric.
                    getUtility = function(arm,timestep) {
                      return (self$utilities[[timestep]][arm])
                    },
                    
                    #' @description
                    #' Returns the ground-truth utility of all arms at a specific timestep.
                    #'
                    #' @param timestep (`integer(1)`)\cr.
                    #' 
                    #'
                    #' @return vector.
                    getUtilities = function(timestep) {
                      return (self$utilities[[timestep]])
                    },
                    
                    #' @description
                    #' Returns the feedback for a specific pair of arms at a specific timestep.
                    #'
                    #' @param arm_pair (`integer(2)`)\cr.
                    #' @param timestep (`integer(1)`)\cr.
                    #' 
                    #'
                    #' @return Boolean.
                    getFeedback = function(arm_pair,timestep) {
                      return (self$pert_utilities[[timestep]][arm_pair[1]]>self$pert_utilities[[timestep]][arm_pair[2]])
                    }
                      
                    
                    
                  ),
                  
                  
                  
)

