
#' @title Performance Class
#' 
#'
#' @description
#' Class to run several runs of learners at once.
#'



Performance = R6Class("Performance",
                   public = list(
                     
                     #' @field learners (`numeric()`)\cr
                     #' List of Lerner instances to be evaluated.
                     learners = NULL,
                     
                     #' @field problem_inst (`list()`)\cr
                     #' A Feedback instance determining the problem environment.
                     problem_inst = list(),
                     
                     #' @field regret_measure (`list()`)\cr
                     #' Function specifying the regret measure.
                     regret_measure = NULL,
                     
                     #' @field reps (`integer()`)\cr
                     #' Number of repetitions.
                     reps = NULL,
                     
                     #' @field cores (`integer()`)\cr
                     #' Number of cores.
                     cores = NULL,
                     
                     #' @field results (`list()`)\cr
                     #' stores the results.
                     results = list(),
                     
                     #' @field result_statistics (`list()`)\cr
                     #' stores the relevant statistics from the results.
                     result_statistics = list(),
                     
                     #' @description
                     #' Creates a new instance of this [R6][R6::R6Class] class.
                     #'
                     initialize = function(learners, d, time_horizon, n, diff, pert_dis ,  regret_measure = mean, reps = 1, cores = 1) {
                       
                       self$learners            = learners
                       self$problem_inst        = list(d = d, time_horizon = time_horizon, n_arms = n, diff = diff, pert_dis = pert_dis)
                       self$regret_measure      = regret_measure
                       self$reps                = reps
                       self$cores               = cores
                       
                     },
                     
                     
                     #' @description
                     #' Printer.
                     #' @param ... (ignored).
                     print = function() { 
                       cat("  diffculty: ", self$problem_inst$diff, "\n", sep = "")
                       cat("  time horizon: ", self$problem_inst$time_horizon, "\n", sep = "")
                       cat("  number of arms: ", self$problem_inst$n_arms, "\n", sep = "")
                       cat("  dimension: ", self$problem_inst$d, "\n", sep = "")
                       cat("  perturbation distribution: ", self$problem_inst$pert_dis, "\n", sep = "")
                     },
                     

                     runSimulationStep = function(rep_i) {
                       
                       context_matrices = list()
                       weights = NULL
                       result_list = list()
                       
                       if (self$problem_inst $d!= 0){ # contextual setting
                         context_matrices = NULL
                         
                         if(self$problem_inst$diff == "easy"){
                           weights = genRandomSampleL2Ring(1,self$problem_inst$d,0,1/sqrt(self$problem_inst$d))
                         }
                         if(self$problem_inst$diff == "medium"){
                           weights = genRandomSampleL2Ring(1,self$problem_inst$d,1/sqrt(self$problem_inst$d),1)
                         }
                         if(self$problem_inst$diff == "hard"){
                           weights = genRandomSampleL2Ring(1,self$problem_inst$d,1,sqrt(self$problem_inst$d))
                         }
                         
                         for (i in 1: (self$problem_inst$time_horizon)){
                           context_matrices[[i]] = matrix(runif(self$problem_inst$n_arms*self$problem_inst$d), ncol=self$problem_inst$n_arms) 
                         }
                         
                         
                       }
                       else{ # non-contextual setting
                         
                       }
					   
                       problem_env = Feedback$new(weights = weights, time_horizon = self$problem_inst$time_horizon, pert_dis = self$problem_inst$pert_dis, context_matrices = context_matrices )
                       
                       temp 					=	 1
					   
                       for (learner in self$learners){
                         learner$clear()
                         learner$setDataModel(problem_env)
                         learner$run(problem_env)
                         result_list[[temp]] 	= list(name = class(learner)[1], cum_regret = cumsum(learner$regret), cum_times = cumsum(learner$times) )
                         temp 					= temp+1
                       }
                       
                       
                       return (result_list)
                     },
                     

                     runSimulations = function() {
                       
                       self$results = mclapply(X = 1:self$reps,self$runSimulationStep, mc.cores = self$cores)
                       
                     },
                     

                     getStatistics = function() {
                       
                       for (j in 1:length(self$learners)){
                          self$result_statistics[[j]] = list(name = "",mean_cum_reg = NULL, mean_cum_time = NULL, sd_cum_reg = NULL, sd_cum_time =NULL, max_reg=NULL)
                         
                          self$result_statistics[[j]]$name = self$results[[1]][[j]][1]
                          
                          temp_reg    = self$results[[1]][[j]]$cum_regret
                          temp_times  = self$results[[1]][[j]]$cum_times
                            
                          for (i in 2:length(self$results)){
                         
                            temp_reg    = rbind(temp_reg,   self$results[[i]][[j]]$cum_regret)
                            temp_times  = rbind(temp_times, self$results[[i]][[j]]$cum_times)
                          }
                          
                          self$result_statistics[[j]]$mean_cum_reg  = apply(temp_reg,2,mean)
                          self$result_statistics[[j]]$mean_cum_time = apply(temp_times,2,mean)
                          self$result_statistics[[j]]$sd_cum_reg    = apply(temp_reg,2,sd)
                          self$result_statistics[[j]]$sd_cum_time   = apply(temp_times,2,sd)
                          self$result_statistics[[j]]$max_reg       = self$result_statistics[[j]]$mean_cum_reg[self$problem_inst$time_horizon]
                          
                       }
                       
                       
                       
                     },
                     

                     plotCumRegret = function() {
             
                       
                       cex_const	=	2.2
                       ylim_min		=	0
                       ylim_max   	=   0
                       for (j in 1:length(self$learners)){
                         ylim_max = max(ylim_max,self$result_statistics[[j]]$max_reg)
                       }
                       
                    
                       Time_Steps = self$problem_inst$time_horizon
                        
                       legend_vec = c(self$result_statistics[[1]]$name)
                       
                       plot(seq(1,Time_Steps,1),self$result_statistics[[1]]$mean_cum_reg,type="l",ylab="",xlab="",cex.axis=cex_const, ylim=c(ylim_min,ylim_max)	)
                       lines(seq(1,Time_Steps,1),self$result_statistics[[1]]$mean_cum_reg+self$result_statistics[[1]]$sd_cum_reg,col=1,lty=2)
                       lines(seq(1,Time_Steps,1),self$result_statistics[[1]]$mean_cum_reg-self$result_statistics[[1]]$sd_cum_reg,col=1,lty=2)
                       title( main=paste(self$problem_inst$diff, ", d = ", self$problem_inst$d, ", n = ", self$problem_inst$n_arms), xlab="T", ylab="",cex.lab=cex_const) 
                       
                       for (j in 2:length(self$learners)){
                         
                         lines(seq(1,Time_Steps,1),self$result_statistics[[j]]$mean_cum_reg,col=j)
                         lines(seq(1,Time_Steps,1),self$result_statistics[[j]]$mean_cum_reg+self$result_statistics[[j]]$sd_cum_reg,col=j,lty=2)
                         lines(seq(1,Time_Steps,1),self$result_statistics[[j]]$mean_cum_reg-self$result_statistics[[j]]$sd_cum_reg,col=j,lty=2)
                         
                       
                       legend_vec = c(legend_vec,self$result_statistics[[j]]$name)
                       }
                       
                       legend("topleft",legend=legend_vec,col=1:length(self$learners),lty=rep(1,length(self$learners)),cex=1.5)
                       
                       
                       
                       
                     },
                     

                     reportRuntimes = function() {
             

                    
                       Time_Steps = self$problem_inst$time_horizon
                       
					   
					   temp = c(self$result_statistics[[1]]$mean_cum_time[Time_Steps], self$result_statistics[[1]]$sd_cum_time[Time_Steps])
					   
                       
                       for (j in 2:length(self$learners)){
                         
							temp = rbind(temp,  c(self$result_statistics[[j]]$mean_cum_time[Time_Steps],self$result_statistics[[j]]$sd_cum_time[Time_Steps]) )
                         
                         
                       }
                       
                       
                       return(temp)
                       
                       
                     }

                   ),
                   
                   
                   
)





